/*
-------------------------------------------------------------------------------
  P h o t o   E x a m p l e   A p p l i c a t i o n

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
       File: "SeriesPersister.java"
    Created: 19.03.2006 (18:56:27)
        $Id: SeriesPersister.java 76 2006-03-29 19:16:10Z Dirk $
  $Revision: 76 $
      $Date: 2006-03-29 21:16:10 +0200 (Mi, 29 Mrz 2006) $
    $Author: Dirk $
===============================================================================
*/

package com.dgrossmann.photo.dir.persist_db;

import java.io.FileInputStream;
import java.io.InputStream;
import java.util.Iterator;

import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import com.dgrossmann.photo.dir.AbstractFSObject;
import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.entity.AbstractFileObject;
import com.dgrossmann.photo.entity.File;
import com.dgrossmann.photo.entity.FileData;
import com.dgrossmann.photo.entity.Group;
import com.dgrossmann.photo.entity.IPhotoDao;
import com.dgrossmann.photo.ui.ExplorerMainFrame;
import com.dgrossmann.photo.webexport.ExportFactory;

/**
 * Class that persists the complete contents of a series into the database.
 * @author Dirk Grossmann
 */
public class SeriesPersister
{
    private static ApplicationContext s_beanFactory =
        new ClassPathXmlApplicationContext(
            "WEB-INF/photo-data-access-beans.xml");

    private ExplorerMainFrame m_explorerFrame;

    /**
     * Creates a new <tt>SeriesPersister</tt> instance.
     * @param explorerFrame - The parent explorer frame that is used only to get
     * the exported files
     */
    public SeriesPersister (ExplorerMainFrame explorerFrame)
    {
        m_explorerFrame = explorerFrame;
    } // SeriesPersister

    /**
     * Saves the content of a series to the datatbase.
     * @param seriesObj - Directory object for the series to save
     * @param bOnlyImages - <tt>True</tt> to save only images; <tt>false</tt>
     * to also save ordinary files, references, and separators
     * @param bOnlyExported - <tt>True</tt> to only consider exported groups
     * and files; <tt>false</tt> to export everything to the database
     * @param bUseMainExportedImage - <tt>True</tt> to use the main exported
     * image if persent; <tt>false</tt> to always use the original image
     */
    public void saveToDatabase
        ( DirectoryObject seriesObj
        , boolean         bOnlyImages
        , boolean         bOnlyExported
        , boolean         bUseMainExportedImage
        )
    {
        Group persSeries = this.createPersistentGroup(seriesObj, null,
        	bOnlyImages, bOnlyExported, bUseMainExportedImage);
        ((IPhotoDao) s_beanFactory.getBean("photoDAO")).createGroup(persSeries);
    } // saveToDatabase

    /**
     * Private method to create a persistent group.
     * @param dirObj - Directory object for the group to save
     * @param series - The associated series group for the group to save
     * @param bOnlyImages - <tt>True</tt> to save only images; <tt>false</tt>
     * to also save ordinary files, references, and separators
     * @param bOnlyExported - <tt>True</tt> to only consider exported groups
     * and files; <tt>false</tt> to export everything to the database
     * @param bUseMainExportedImage - <tt>True</tt> to use the main exported
     * image if persent; <tt>false</tt> to always use the original image
     * @return The group object
     */
    private Group createPersistentGroup
        ( DirectoryObject dirObj
        , Group           series
        , boolean         bOnlyImages
        , boolean         bOnlyExported
        , boolean         bUseMainExportedImage
        )
    {
        Iterator        iter;
        DirectoryObject subDirObj;
        FileObject      fileObj;
        Group           group, subGroup, prevSubGroup;
        File            file, prevFile;

        // Prepare the group itself.
        group = new Group();
        if (series == null)
        	series = group;
        this.fillGroupDataFrom(group, dirObj);
        group.setSeries(series);
        // Prepare the files.
        prevFile = null;
        iter = dirObj.getFileIterator();
        while (iter.hasNext())
        {
            fileObj = (FileObject) iter.next();
            if (bOnlyImages && fileObj.getFileType() != FileObject.TYPE_IMAGE &&
                fileObj.getFileType() != FileObject.TYPE_IMAGE_PREVIEW)
            {
                continue;
            }
            if (bOnlyExported && !fileObj.isToExport())
                continue;
            file = new File();
            this.fillFileDataFrom(file, fileObj, bUseMainExportedImage);
            group.addFile(file, prevFile);
            prevFile = file;
        }
        // Prepare the sub groups.
        prevSubGroup = null;
        iter = dirObj.getSubDirIterator();
        while (iter.hasNext())
        {
            subDirObj = (DirectoryObject) iter.next();
            if (bOnlyExported && !subDirObj.isToExport())
                continue;
            subGroup = this.createPersistentGroup(subDirObj, series,
            	bOnlyImages, bOnlyExported, bUseMainExportedImage);
            group.addSubGroup(subGroup, prevSubGroup);
            prevSubGroup = subGroup;
        }
        return group;
    } // createPersistentGroup

    /**
     * Private method to fill an abstract file entity with the data from a file
     * system object.
     * @param fo - The abstract file entity
     * @param fsObj - The file system object
     */
    private void fillAbstractFileDataFrom
        ( AbstractFileObject fo, AbstractFSObject fsObj)
    {
        fo.setName(fsObj.getFileName());
        fo.setTitle(fsObj.getTitle(true));
        fo.setSubtitle(fsObj.get(AbstractFSObject.SUBTITLE));
        fo.setDescription(fsObj.get(AbstractFSObject.DESCRIPTION));
        fo.setPhotoLocation(fsObj.get(AbstractFSObject.LOCATION));
        fo.setExport(new Boolean(fsObj.isToExport()));
    } // fillAbstractFileDataFrom

    /**
     * Private method to fill a group with the data from a directory object.
     * @param group - The group to fill
     * @param dirObj - The source JPhoto-Explorer directory object
     */
    private void fillGroupDataFrom (Group group, DirectoryObject dirObj)
    {
        // Fill the common information.
        this.fillAbstractFileDataFrom(group, dirObj);
        // Fill the directory-specific data.
        group.setBeginDate(dirObj.get(AbstractFSObject.DATE_BEGIN));
        group.setEndDate(dirObj.get(AbstractFSObject.DATE_END));
    } // fillGroupDataFrom

    /**
     * Private method to fill a file with the data from a file object. This
     * method also creates and associates the file data object.
     * @param file - The persistent file to fill
     * @param fileObj - The source JPhoto-Explorer file object
     * @param bUseMainExportedImage
     */
    private void fillFileDataFrom
        ( File       file
        , FileObject fileObj
        , boolean    bUseMainExportedImage
        )
    {
        java.io.File   mainFile, indexFile;
        java.io.File[] exportedFiles;
        FileData       fileData;

        // Fill the common information.
        this.fillAbstractFileDataFrom(file, fileObj);
        // Override the file name if we use the exported image.
        if (bUseMainExportedImage)
            file.setName(fileObj.getBaseFileName() + ".jpg");
        // Set the file type.
        if (fileObj.isReference())
            file.setFileType(File.FILETYPE_REFERENCE);
        else if (fileObj.isSeparator())
            file.setFileType(File.FILETYPE_SEPARATOR);
        else if (fileObj.getFileType() == FileObject.TYPE_IMAGE_PREVIEW)
            file.setFileType(File.FILETYPE_IMAGE);
        else
            file.setFileType(File.FILETYPE_FILE);
        // Fill the file-specific data.
        file.setExpDate(fileObj.get(AbstractFSObject.DATE_BEGIN));
        file.setHref(fileObj.get(AbstractFSObject.HREF));
        // Create the file data.
        mainFile = new java.io.File(fileObj.getFullPath());
        indexFile = null;
        if (file.getFileType() == File.FILETYPE_IMAGE || mainFile.exists())
        {
        	// Get the exported files.
            exportedFiles = ExportFactory.getExport(
                m_explorerFrame.getSettings(),
                m_explorerFrame.getSeriesContainer(), m_explorerFrame).
                getExportedFiles(fileObj);
            if (exportedFiles != null)
            {
            	if (bUseMainExportedImage && exportedFiles[0] != null &&
                    exportedFiles[0].exists())
            	{
            		mainFile = exportedFiles[0];
            	}
            	if (exportedFiles[1] != null && exportedFiles[1].exists())
            		indexFile = exportedFiles[1];
            }
            // Create the FileData for the main and the index file.
            if (mainFile.exists())
            {
            	fileData = this.createFileData(mainFile);
            	if (fileData != null)
            		file.setFileData(fileData);
            }
            if (indexFile != null && indexFile.exists())
            {
            	fileData = this.createFileData(indexFile);
            	if (fileData != null)
            		file.setIndexFileData(fileData);
            }
        }
    } // fillFileDataFrom

    /**
     * Private method to create a FileData object from a file.
     * @param f - The image file
     * @return The file data object
     */
    private FileData createFileData (java.io.File f)
    {
    	FileData    fileData;
        String      name;
        InputStream is;
        byte[]      fileContent;
        int         fileContentLength;

    	fileData = new FileData();
    	name = f.getAbsolutePath().toLowerCase();
    	if (name.endsWith(".png"))
    		fileData.setMimeType("image/ping");
    	else if (name.endsWith(".gif"))
    		fileData.setMimeType("image/gif");
    	else if (name.endsWith(".jpg") || name.endsWith(".jpeg"))
    		fileData.setMimeType("image/jpeg");
    	else if (name.endsWith(".tif") || name.endsWith(".tiff"))
    		fileData.setMimeType("image/tiff");
    	try
    	{
    		is = new FileInputStream(f);
    		fileContentLength = (int) f.length();
    		fileContent = new byte[fileContentLength];
    		is.read(fileContent, 0, fileContentLength);
    		is.close();
    		fileData.setData(fileContent);
    		fileData.setFileSize(new Long(fileContentLength));
    		return fileData;
    	}
    	catch (Exception exc)
    	{
    		return null;
    	}
    }
} // SeriesPersister
