/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: DirectoryObject
    Created: 2 January, 2003
        $Id: DirectoryObject.java 158 2009-05-06 19:49:13Z dirk $
  $Revision: 158 $
      $Date: 2009-05-06 21:49:13 +0200 (Mi, 06 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.dir;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;

/**
 * Instances represent directories in the file system.
 */
public class DirectoryObject extends AbstractFSObject
{
    private ArrayList<DirectoryObject> m_subDirectories;
    private ArrayList<FileObject>      m_files;
    private boolean                    m_isChanged;
    private boolean                    m_isError;

    /**
     * Constructor of a <tt>DirectoryObject</tt> instance.
     * @param name - Name of the directory in the file system
     * @param parentDirObj - Parent directory of <tt>null</tt> for a series
     * directory.
     */
    public DirectoryObject (String name, DirectoryObject parentDirObj)
    {
        super(parentDirObj);
        this.setFileName(name, false);
        // Set the instance variables.
        m_subDirectories = new ArrayList<DirectoryObject>(10);
        m_files = new ArrayList<FileObject>(40);
        m_isChanged = m_isError = false;
    } // DirectoryObject

    /**
     * @see com.dgrossmann.photo.dir.AbstractFSObject#isEmpty()
     */
    @Override
	public boolean isEmpty ()
    {
        File dir = new File(this.getFullPath());
        if (!dir.isDirectory() || dir.list() == null)
            return true;
        return false;
    } // isEmpty

    /**
     * Gets the directory name.
     * @return The directory name
     */
    @Override
	public String getFileName ()
    {
        String name = super.getFileName();
        if (name.length() == 0 || this.getParent() != null)
            return name;
        // For series directories, we return the last path element.
        int i = name.lastIndexOf(File.separator);
        if (i < 0)
            i = name.lastIndexOf("/");
        return (i >= 0 && i < name.length()-1) ? name.substring(i + 1) : name;
    } // getFileName

    /**
     * Gets the number of sub directories.
     * @return The number of sub directories
     */
    public int getSubDirCount ()
    {
        return m_subDirectories.size();
    } // getSubDirCount

    /**
     * Gets the sub directory at the specified index.
     * @param i - <tt>0</tt>-based index
     * @return The sub directory object
     */
    public DirectoryObject getSubDirAt (int i)
    {
        if (i < 0 || i >= m_subDirectories.size())
            return null;
        return m_subDirectories.get(i);
    } // getSubDirAt

    /**
     * Gets the index of a sub directory.
     * @param subDirObj - Sub directory object
     * @return The <tt>0</tt>-based index in the parent directory
     */
    public int getIndexOfSubDir (DirectoryObject subDirObj)
    {
        return m_subDirectories.indexOf(subDirObj);
    } // getIndexOfSubDir

    /**
     * Gets a subdirectory iterator.
     * @return The subdirectory iterator
     */
    public Iterator<DirectoryObject> getSubDirIterator ()
    {
        return m_subDirectories.iterator();
    } // getSubDirIterator

    /**
     * Gets the number of files contained in this directory.
     * @return The number of files
     */
    public int getFileCount ()
    {
        return m_files.size();
    } // getFileCount

    /**
     * Gets the file at the specified index.
     * @param i - <tt>0</tt>-based index
     * @return The corresponding file object
     */
    public FileObject getFileAt (int i)
    {
        if (i < 0 || i >= m_files.size())
            return null;
        return m_files.get(i);
    } // getFileAt

    /**
     * Gets the index of a file object.
     * @param fileObj - The file object
     * @return <tt>0</tt>-based index
     */
    public int getIndexOfFile (FileObject fileObj)
    {
        return m_files.indexOf(fileObj);
    } // getIndexOfFile

    /**
     * Gets a file iterator.
     * @return The file iterator
     */
    public Iterator<FileObject> getFileIterator ()
    {
        return m_files.iterator();
    } // getFileIterator

    /**
     * Adds a file or subdirectory entry to this instance at the specified
     * <tt>0</tt>-based index.
     * @param fsObj - File or directory object to add
     * @param index - Index to add into the file/subdirectory list or
     * <tt>-1</tt> to append at the end.
     * @return <tt>True</tt> if successful
     */
    public boolean addChild (AbstractFSObject fsObj, int index)
    {
        DirectoryObject oldParent = null;
        int             oldIndex  = -1;
        File            oldFile   = null;

        DirectoryObject d = (fsObj instanceof DirectoryObject) ?
            ((DirectoryObject) fsObj) : null;
        FileObject f = (fsObj instanceof FileObject) ?
            ((FileObject) fsObj) : null;
        if (d == null && f == null)
            return false;
        // Remove the object from the old parent directory.
        oldParent = (DirectoryObject) fsObj.getParent();
        if (oldParent != null)
        {
            if (fsObj.getFileName().length() > 0)
            {
                // Remember the file associated with this object.
                if (d != null)
                    oldIndex = oldParent.getIndexOfSubDir(d);
                else
                    oldIndex = oldParent.getIndexOfFile(f);
                oldFile = new File(fsObj.getFullPath());
            }
            if (d != null)
                oldParent.removeSubDir(d);
            else
                oldParent.removeFile(f);
        }
        // add the file to us.
        fsObj.m_parent = this;
        if (d != null)
        {
            if (index < 0 || index > m_subDirectories.size())
                m_subDirectories.add(d);
            else
                m_subDirectories.add(index, d);
        }
        else
        {
            if (index < 0 || index > m_files.size())
                m_files.add(f);
            else
                m_files.add(index, f);
        }
        this.setChanged();
        // Move the file associated with this object.
        if (oldFile != null)
        {
            if (oldFile.renameTo(new File(fsObj.getFullPath())))
                return true;
            // Error: Revert to the old state.
            fsObj.m_parent = null;
            if (d != null)
                this.removeSubDir(d);
            else
                this.removeFile(f);
            oldParent.addChild(fsObj, oldIndex);
            return false;
        }
        return true;
    } // addChild

    /**
     * Removes a subdirectory.
     * @param dirObj - Subdirectory to remove
     */
    public void removeSubDir (DirectoryObject dirObj)
    {
        this.setChanged();
        m_subDirectories.remove(dirObj);
    } // removeSubDir

    /**
     * Removes a file.
     * @param fileObj - File to remove
     */
    public void removeFile (FileObject fileObj)
    {
        this.setChanged();
        m_files.remove(fileObj);
    } // removeFile

    /**
     * Checks whether the object can be moved up.
     * @param fsObj - The file system object to check
     * @return <tt>True</tt> if the operation can be performed
     */
    public boolean canMoveUp (AbstractFSObject fsObj)
    {
        int index;
        if (fsObj instanceof DirectoryObject)
        {
            index = m_subDirectories.indexOf(fsObj);
            if (index <= 0 || index >= m_subDirectories.size())
                return false;
            return true;
        }
        else if (fsObj instanceof FileObject)
        {
            index = m_files.indexOf(fsObj);
            if (index <= 0 || index >= m_files.size())
                return false;
            return true;
        }
        return false;
    } // canMoveUp

    /**
     * Checks whether the object can be moved down.
     * @param fsObj - The file system object to check
     * @return <tt>True</tt> if the operation can be performed
     */
    public boolean canMoveDown (AbstractFSObject fsObj)
    {
        int index;
        if (fsObj instanceof DirectoryObject)
        {
            index = m_subDirectories.indexOf(fsObj);
            if (index < 0 || index >= m_subDirectories.size() - 1)
                return false;
            return true;
        }
        else if (fsObj instanceof FileObject)
        {
            index = m_files.indexOf(fsObj);
            if (index < 0 || index >= m_files.size() - 1)
                return false;
            return true;
        }
        return false;
    } // canMoveDown

    /**
     * Move up one file.
     * @param fileObj - The file object to move up
     */
    public void moveFileUp (FileObject fileObj)
    {
        int index = m_files.indexOf(fileObj);
        if (index <= 0 || index >= m_files.size())
            return;
        FileObject eBefore = m_files.get(index - 1);
        m_files.set(index-1, fileObj);
        m_files.set(index, eBefore);
        this.setChanged();
    } // moveFileUp

    /**
     * Move down one file.
     * @param fileObj - The file object to move down
     */
    public void moveFileDown (FileObject fileObj)
    {
        int index = m_files.indexOf(fileObj);
        if (index < 0 || index >= m_files.size() - 1)
            return;
        FileObject eAfter = m_files.get(index + 1);
        m_files.set(index+1, fileObj);
        m_files.set(index, eAfter);
        this.setChanged();
    } // moveFileDown

    /**
     * Move up one subdirectory.
     * @param dirObj - The subdirectory to move up
     */
    public void moveSubDirUp (DirectoryObject dirObj)
    {
        int index = m_subDirectories.indexOf(dirObj);
        if (index <= 0 || index >= m_subDirectories.size())
            return;
        DirectoryObject eBefore = m_subDirectories.get(index - 1);
        m_subDirectories.set(index-1, dirObj);
        m_subDirectories.set(index, eBefore);
        this.setChanged();
    } // moveSubDirUp

    /**
     * Move down one subdirectory.
     * @param dirObj - The subdirectory to move down
     */
    public void moveSubDirDown (DirectoryObject dirObj)
    {
        int index = m_subDirectories.indexOf(dirObj);
        if (index < 0 || index >= m_subDirectories.size() - 1)
            return;
        DirectoryObject eAfter = m_subDirectories.get(index + 1);
        m_subDirectories.set(index+1, dirObj);
        m_subDirectories.set(index, eAfter);
        this.setChanged();
    } // moveSubDirDown

    /**
	 * Reorders the files contained in this directory such that they are
	 * ordered by file name.
	 */
	public void orderFilesByFileName ()
	{
		Collections.sort(m_files, new Comparator<FileObject>() {
			public int compare (FileObject file1, FileObject file2)
			{
				if (file1 == file2)
					return 0;
				if (file1 == null)
					return (file2 == null) ? 0 : -1;
				if (file2 == null)
					return 1;
				if (file1.getFileName() == null)
					return (file2.getFileName() == null) ? 0 : -1;
				return file1.getFileName().compareTo(file2.getFileName());
			}
		});
	} // orderFilesByFileName

	/** Clears file and directory information. */
    public void clear ()
    {
        m_subDirectories.clear();
        m_files.clear();
        this.setChanged();
    } // clear

    /** Marks this directory as changed. */
    public void setChanged ()
    {
        if (this.getParent() != null)
            ((DirectoryObject) this.getParent()).setChanged();
        else
            m_isChanged = true;
    } // setChanged

    /** Marks this directory as saved (unchanged). */
    public void setSaved ()
    {
        if (this.getParent() != null)
            ((DirectoryObject) this.getParent()).setChanged();
        else
            m_isChanged = false;
    } // setSaved

    /**
     * Checks whether this series is changed.
     * @return <tt>True</tt> if changed
     */
    public boolean isChanged ()
    {
        DirectoryObject dirObj = this;
        while (dirObj.getParent() != null)
            dirObj = (DirectoryObject) dirObj.getParent();
        return dirObj.m_isChanged;
    } // isChanged

    /** Marks this directory as erroneous (should not be saved). */
    public void setError ()
    {
        if (this.getParent() != null)
            ((DirectoryObject) this.getParent()).setChanged();
        else
            m_isError = true;
    } // setError

    /**
     * Checks whether this series is erroneous.
     * @return <tt>Treu</tt> iff erroneous
     */
    public boolean isError ()
    {
        DirectoryObject dirObj = this;
        while (dirObj.getParent() != null)
            dirObj = (DirectoryObject) dirObj.getParent();
        return dirObj.m_isError;
    } // isError

    /**
     * Private Helper method for delete.
     * @param dirPath - Path denoting the directory to delete
     * @param bForce - <tt>True</tt> to force deletion of non-empty directories
     * @return <tt>True</tt> if successful
     */
    private boolean deleteDir (String dirPath, boolean bForce)
    {
        File thisFile, f;
        int  i;

        thisFile = new File(dirPath);
        // Delete the sub objects.
        String[] subObjNames = thisFile.list();
        if (subObjNames == null)
            return true;
        for (i = 0; i < subObjNames.length; i++)
        {
            if (subObjNames[i].equals(".") || subObjNames[i].equals(".."))
                continue;
            f = new File(dirPath + File.separator + subObjNames[i]);
            if (f.isFile())
            {
                if (!bForce || !f.delete())
                    return false;
            }
            else if (f.isDirectory())
            {
                if (!this.deleteDir(dirPath + File.separator
                    + subObjNames[i], bForce))
                {
                    return false;
                }
            }
            else
                return false;
        }
        return thisFile.delete();
    } // deleteDir

    /**
     * Deletes this instance and all files and subdirectories from the file
     * system
     * @param bForce - <tt>True</tt> to force deletion even if the directories
     * are not empty. Empty means that the directory and its substructure may
     * contain directories, but no files.
     * @return <tt>True</tt> on success, and <tt>false</tt> on failure
     */
    public boolean delete (boolean bForce)
    {
        boolean bOK;

        // Clear the sub object information.
        m_subDirectories.clear();
        m_files.clear();
        // Delete the sub objects.
        bOK = this.deleteDir(this.getFullPath(), bForce);
        // Finish deletion.
        if (bOK)
        {
            if (m_parent != null)
                ((DirectoryObject) m_parent).removeSubDir(this);
            m_fileName = "";
            this.setChanged();
        }
        return bOK;
    } // delete
} // DirectoryObject
