/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: FileObject
    Created: 2 January, 2003
        $Id: FileObject.java 160 2009-05-31 07:57:29Z dirk $
  $Revision: 160 $
      $Date: 2009-05-31 09:57:29 +0200 (So, 31 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.dir;

import java.io.File;

import com.dgrossmann.photo.webexport.ImageMagickExporter;

/**
 * Represents files in the file system, or references or separators (without a
 * counterpart in the file system structure).
 */
public class FileObject extends AbstractFSObject
{
    public static final int TYPE_UNKNOWN       =  0;
    public static final int TYPE_IMAGE_PREVIEW =  1;
    public static final int TYPE_IMAGE         =  2;
    public static final int TYPE_HTML_DOC      =  3;
    public static final int TYPE_XML_DOC       =  4;
    public static final int TYPE_PDF_DOC       =  5;
    public static final int TYPE_MSWORD_DOC    =  6;
    public static final int TYPE_MSVISIO_DOC   =  7;
    public static final int TYPE_TEXT_DOC      =  8;
    public static final int TYPE_CAD_DOC       =  9;
    public static final int TYPE_ZIP_DOC       = 10;
    public static final int TYPE_REFERENCE     = 11;
    public static final int TYPE_HTML_REF      = 12;
    public static final int TYPE_SEPARATOR     = 13;

    public static final String SEPARATOR = "separator:";

    /**
     * Creates a new <tt>FileObject</tt> instance.
     * @param name - Name of the containing directory in the file system
     * @param parentDir - Parent directory object
     */
    public FileObject (String name, DirectoryObject parentDir)
    {
        super(parentDir);
        this.setFileName(name, false);
        m_isReference = (m_fileName.length() == 0);
    } // FileObject

    /**
	 * If this file is an image, set the properties from the image meta data.
	 */
	public void setPropertiesFromImageMetadata ()
	{
		/*
		String           fileName;
		Metadata         metadata;
		Iterator<Object> tags;
		Directory        directory;
		Tag              tag;

		try
		{
			fileName = this.getFileName().toLowerCase();
			if (fileName.endsWith("jpg") || fileName.endsWith("jpeg"))
			{
				metadata = JpegMetadataReader.readMetadata(
					new File(this.getFullPath()));
			}
			else if (fileName.endsWith("tif") || fileName.endsWith("tiff"))
			{
				metadata = TiffMetadataReader.readMetadata(
					new File(this.getFullPath()));
			}
			else
				return;
			if (!metadata.containsDirectory(IptcDirectory.class))
				return;
			directory = metadata.getDirectory(IptcDirectory.class);
			// Iterate through tags and print to System.out
			tags = directory.getTagIterator();
			while (tags.hasNext())
			{
				tag = (Tag) tags.next();
				System.out.println(tag);
			}
			// TODO Auto-generated method stub
		}
		catch (Throwable exc)
		{
			System.err.println("File \"" + this.getFullPath() + "\":");
			System.err.println("     Cannot read image meta data - " +
				exc.toString());
		}
		*/
	} // setPropertiesFromImageMetadata

	/**
     * @see com.dgrossmann.photo.dir.AbstractFSObject#isEmpty()
     */
    @Override
	public boolean isEmpty ()
    {
        if (this.isSeparator())
            return false;
        return super.isEmpty();
    } // isEmpty

    /**
     * Tests whether this instance is a separator.
     * @return <tt>True</tt> iff this instance is a separator
     */
    public boolean isSeparator ()
    {
        if (m_fileNamePart.length() == 0)
            return false;
        if (m_fileNamePart.equalsIgnoreCase(SEPARATOR))
            return true;
        return false;
    } // isSeparator

    /** Makes this instance a separator. */
    public void makeSeparator ()
    {
        this.setFileNamePart(SEPARATOR);
        m_properties.put(AbstractFSObject.HREF, "");
        this.setFileName("", false);
        m_isReference = true;
    } // makeSeparator

    /**
     * Gets the base name of this file without extension.
     * @return Base name string
     */
    public String getBaseFileName ()
    {
        String name = this.getFileName();
        int i = name.indexOf('.');
        if (i >= 0)
            return name.substring(0, i);
        return name;
    } // getBaseFileName

    /**
     * Gets the file type.
     * @return File type
     */
    public int getFileType ()
    {
        String str;

        if (this.isSeparator())
            return TYPE_SEPARATOR;
        if (this.isReference())
        {
            str = this.get(AbstractFSObject.HREF);
            if (str != null)
            {
                str = str.toLowerCase().trim();
                if (str.startsWith("http:") ||
                    str.startsWith("ftp:"))
                {
                    return TYPE_HTML_REF;
                }
            }
            return TYPE_REFERENCE;
        }
        str = this.getFileName().toLowerCase();
        if (str.endsWith("gif") || str.endsWith("jpg") ||
             str.endsWith("jpeg") || str.endsWith("png"))
        {
            return TYPE_IMAGE_PREVIEW;
        }
        if (str.endsWith("tif") || str.endsWith("tiff"))
        {
            return ImageMagickExporter.isImageMagickAvailable() ?
                TYPE_IMAGE_PREVIEW : TYPE_IMAGE;
        }
        if (str.endsWith("psd") || str.endsWith("xcf") ||
            str.endsWith("bmp") || str.endsWith("pcx"))
        {
            return TYPE_IMAGE;
        }
        if (str.endsWith("htm") || str.endsWith("html"))
            return TYPE_HTML_DOC;
        if (str.endsWith("xml") || str.endsWith("dtd") || str.endsWith("xsd"))
            return TYPE_XML_DOC;
        if (str.endsWith("pdf"))
            return TYPE_PDF_DOC;
        if (str.endsWith("doc"))
            return TYPE_MSWORD_DOC;
        if (str.endsWith("vsd"))
            return TYPE_MSVISIO_DOC;
        if (str.endsWith("txt") || str.endsWith("text") || str.endsWith("tex"))
            return TYPE_TEXT_DOC;
        if (str.endsWith("dxf") || str.endsWith("dwg"))
            return TYPE_CAD_DOC;
        if (str.endsWith("zip") || str.endsWith("tar") || str.endsWith("gz") ||
            str.endsWith("tgz"))
        {
            return TYPE_ZIP_DOC;
        }
        return TYPE_UNKNOWN;
    } // getFileType

    /**
     * Returns <tt>true</tt> if ImageMagick is required for displaying the
     * image represented by this instance.
     * @return <tt>True</tt> if ImageMagick is required
     */
    public boolean isImageMagickRequiredForDisplay ()
    {
        String str = this.getFileName().toLowerCase();
        if (str.endsWith("tif") || str.endsWith("tiff"))
        {
            return true;
        }
        return false;
    } // isImageMagickRequiredForDisplay

    /**
     * @see com.dgrossmann.photo.dir.AbstractFSObject#getFileSizeStr(boolean)
     */
    @Override
	public String getFileSizeStr (boolean bInKB)
    {
        long sz = (new File(this.getFullPath())).length();
        if (bInKB && sz >= 1024)
        {
            double szInKB = sz / 1024.0;
            String suff = " KB";
            int prec = 2;
            if (szInKB > 1024.0)
            {
                szInKB /= 1024.0;
                suff = " MB";
                prec = 3;
            }
            String s = Double.toString(szInKB);
            int    i = s.indexOf('.');
            if (i > 0 && i + prec < s.length())
                s = s.substring(0, i + prec);
            if (s.endsWith(".0") || s.endsWith(".00"))
                s = s.substring(0, i);
            return s + suff;
        }
        return Long.toString(sz) + " Bytes";
    } // getFileSizeStr

    /**
     * Returns a string representation of this instance.
     * @return String representation (file name or line for separators)
     */
    @Override
	public String toString ()
    {
        if (this.isSeparator())
            return "-------";
        return super.toString();
    } // toString
} // FileObject
