/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: SeriesContainer
    Created: 2 January, 2003
        $Id: SeriesContainer.java 158 2009-05-06 19:49:13Z dirk $
  $Revision: 158 $
      $Date: 2009-05-06 21:49:13 +0200 (Mi, 06 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.dir;

import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import com.dgrossmann.photo.dir.persist.PersistException;
import com.dgrossmann.photo.dir.persist.PersistFactory;
import com.dgrossmann.photo.settings.Settings;

/**
 * Represents a directory structure that consists of one or more <i>series
 * directories</i> with their substructure (groups, images, and other files).
 */
public class SeriesContainer
{
    private Settings              m_settings;
    private List<String>          m_seriesContainerNames;
    private List<DirectoryObject> m_seriesDirectories;
    private List<DirectoryObject> m_loadedSeriesDirectories;

    /**
     * Creates a new <tt>SeriesContainer</tt> instance. The instance data is
     * accessed using the settings object passed as parameter.
     * @param settings - Settings object
     */
    public SeriesContainer (Settings settings)
    {
        m_settings = settings;
        m_seriesContainerNames = new ArrayList<String>(25);
        m_seriesDirectories = new ArrayList<DirectoryObject>(25);
        m_loadedSeriesDirectories = new ArrayList<DirectoryObject>(25);
        // Load the series containers.
        try
        {
            Iterator<String> iter = m_settings.getSeriesContainerNameIterator();
            while (iter.hasNext())
                this.addSeriesContainer(iter.next());
        }
        catch (PersistException exc)
        {
        }
    } // SeriesContainer

    /**
     * Clears the list of series containers.
     */
    public void clearSeriesContainers ()
    {
        m_seriesContainerNames.clear();
        m_seriesDirectories.clear();
        m_loadedSeriesDirectories.clear();
    } // clearSeriesContainers

    /**
     * Returns whether this instance has series directories.
     * @return <tt>True</tt> iff this instance has series directories
     */
    public boolean hasSeriesDirectories ()
    {
        return m_seriesDirectories.size() > 0;
    } // hasSeriesDirectories

    /**
     * Checks whether a directory object is a series directory.
     * @param dirObj - Directory to be checked
     * @return <tt>True</tt> if <tt>dirObj</tt> is a series directory
     */
    public boolean isSeriesDirectory (DirectoryObject dirObj)
    {
        return m_seriesDirectories.contains(dirObj);
    } // isSeriesDirectory

    /**
     * Private method to get the series from a container directory.
     * @param containerDirName - Name of the container directory
     * @throws PersistException on error
     */
    private void getSeriesFromContainer (String containerDirName)
        throws PersistException
    {
        DirectoryObject de;
        File            dir, f;
        String[]        dirNames;
        String          str;
        int             i;

        dir = new File(containerDirName);
        // Read the series container directory and create a series object for
        // each sub directory.
        dirNames = dir.list();
        if (dirNames == null)
            return;
        for (i = 0; i < dirNames.length; i++)
        {
            str = dirNames[i].toLowerCase();
            if (str.startsWith(".") || str.startsWith("_") ||
                str.startsWith("x_") || str.startsWith("x-"))
            {
                continue;
            }
            f = new File(dir, dirNames[i]);
            if (!f.isDirectory())
                continue;
            de = new DirectoryObject(f.getAbsolutePath(), null);
            m_seriesDirectories.add(de);
        }
    } // getSeriesFromContainer

    /**
     * Merges the list of series container names with the series container names
     * held in this instance.
     * @param seriesContainerNames - The new series container names
     * @throws PersistException
     */
    public void mergeSeriesContainers (List<String> seriesContainerNames)
        throws PersistException
    {
        List<String>     toDelete;
        Iterator<String> iter;
        String           name;

        toDelete = new ArrayList<String>(m_seriesContainerNames.size());
        iter = m_seriesContainerNames.iterator();
        while (iter.hasNext())
        {
            name = iter.next();
            if (!seriesContainerNames.contains(name))
                toDelete.add(name);
        }
        m_seriesContainerNames.removeAll(toDelete);
        iter = seriesContainerNames.iterator();
        while (iter.hasNext())
        {
            name = iter.next();
            if (!m_seriesContainerNames.contains(name))
                this.addSeriesContainer(name);
        }
    } // mergeSeriesContainers

    /**
     * Adds a series directory container. This is a directories containing
     * series directories.
     * @param containerDirName - Path of the directory to be added. Can be
     * absolute or relative to the current directory.
     * @throws PersistException if <tt>containerDirName</tt> is not a
     * directory
     */
    public void addSeriesContainer (String containerDirName)
        throws PersistException
    {
        File dir = new File(containerDirName);
        if (!dir.isDirectory())
        {
            throw new PersistException(PersistException.E_ADD_CONTAINER,
                containerDirName, null, "Not a directory");
        }
        m_seriesContainerNames.add(containerDirName);
        this.getSeriesFromContainer(containerDirName);
    } // addSeriesContainer

    /**
     * Ensures that the <tt>seriesDir</tt> series is loaded from the file
     * system and the meta data.
     * @param seriesDir - Series directory object
     */
    public void ensureLoaded (DirectoryObject seriesDir)
    {
        if (m_loadedSeriesDirectories.contains(seriesDir))
            return;
        try
        {
            this.loadSeries(seriesDir);
        }
        catch (Exception e)
        {
            return;
        }
        m_loadedSeriesDirectories.add(seriesDir);
    } // ensureLoaded

    /**
     * Refreshes the series container contents.
     * @throws PersistException on error
     */
    public void refresh ()
        throws PersistException
    {
        m_seriesDirectories.clear();
        m_loadedSeriesDirectories.clear();
        Iterator<String> iter = m_seriesContainerNames.iterator();
        while (iter.hasNext())
            this.getSeriesFromContainer(iter.next());
    } // refresh

    /**
     * Computes the relative path from file system object named "from" to "to".
     * @param from - From directory name
     * @param to - To file/directory name
     * @return Relative path, using File.separator as separator, or
     * <tt>null</tt> if there is no relative path
     */
    public String getRelativePath (String from, String to)
    {
        StringTokenizer fromDiff;
        String          retPath;
        int             numDiffFrom, i;

        // Normalize the path strings.
        if (File.separatorChar != '\\')
        {
            from = from.replace('\\', File.separatorChar);
            to = to.replace('\\', File.separatorChar);
        }
        if (File.separatorChar != '/')
        {
            from = from.replace('/', File.separatorChar);
            to = to.replace('/', File.separatorChar);
        }
        // Get the prefix directory.
        i = 0;
        while (i < from.length()
            && i < to.length()
            && Character.toLowerCase(from.charAt(i))
                == Character.toLowerCase(to.charAt(i)))
        {
            i++;
        }
        // Cover the cases when "i" equals the string length.
        if (i >= from.length())
        {
            if (i >= to.length())
            {
                // "From" equals "to".
                return "";
            }
            if (to.charAt(i) == File.separatorChar)
            {
                // "From" is a prefix of "to".
                return to.substring(i+1);
            }
            i--;
        }
        // If i >= to.length(), we do not have to do anything: if from[i] is
        // the directory separator, the while loop below does nothing;
        // otherwise it correctly goes back because then the last directories
        // in the paths differ. So, we go back to the next separator.
        while (i >= 0 && from.charAt(i) != File.separatorChar)
            i--;
        if (i <= 0)
        {
            // There is to relative path.
            return null;
        }
        i++;
        // "i" points at the beginning of the first different directory in
        // the paths.
        fromDiff = new StringTokenizer(from.substring(i), File.separator);
        numDiffFrom = fromDiff.countTokens();
        if (!((new File(from)).isDirectory()))
            numDiffFrom--;
        retPath = "";
        while (numDiffFrom > 0)
        {
            retPath += ".." + File.separatorChar;
            numDiffFrom--;
        }
        if (i >= to.length())
        {
            if (retPath.length() > 0 && retPath.endsWith(File.separator))
                return retPath.substring(0, retPath.length()-1);
            return retPath;
        }
        return retPath + to.substring(i);
    } // getRelativePath

    /**
     * Gets the title of this series structure instance.
     * @return Series structure title
     */
    public String getTitle ()
    {
        return m_settings.get(Settings.GALLERY_TITLE);
    } // getTitle

    /**
     * Gets the description of this series structure instance.
     * @param bAsHTML - <tt>True</tt> to expand all empty lines to HTML
     * paragraph start tags.
     * @param pClass - CSS class name for the &lt;p&gt; tags to be inserted
     * @return Series structure description
     */
    public String getDescription (boolean bAsHTML, String pClass)
    {
        String desc = m_settings.get(Settings.GALLERY_DESCRIPTION);
        if (desc.length() == 0)
            return "";
        if (bAsHTML)
        {
            if (pClass != null && pClass.length() > 0)
                pClass = " class=\"" + pClass + "\"";
            else
                pClass = "";
            desc = desc.replaceAll("\\n\\n+", "<p" + pClass + ">");
        }
        return desc;
    } // getDescription

    /**
     * Gets the series directories of this series container structure instance.
     * @return Array of the series directory entry objects
     */
    public DirectoryObject[] getSeriesDirectories()
    {
        List<DirectoryObject>     inv;
        Iterator<DirectoryObject> dirIter;
        DirectoryObject[]         entries;
        DirectoryObject           series;
        File                      f;
        int                       i;

        if (m_seriesDirectories.size() == 0)
            return new DirectoryObject[0];
        inv = new ArrayList<DirectoryObject>(10);
        dirIter = m_seriesDirectories.iterator();
        while (dirIter.hasNext())
        {
            series = dirIter.next();
            f = new File(series.getFullPath());
            if (!f.isDirectory())
                inv.add(series);
        }
        dirIter = inv.iterator();
        while (dirIter.hasNext())
            m_seriesDirectories.remove(dirIter.next());
        entries = new DirectoryObject[m_seriesDirectories.size()];
        i = 0;
        dirIter = m_seriesDirectories.iterator();
        while (dirIter.hasNext())
            entries[i++] = dirIter.next();
        return entries;
    } // getSeriesDirectories

    /**
     * Loads the series where <tt>dirObj</tt> is part of.
     * @param dirObj - The directory object
     * @throws PersistException on load error
     */
    public void loadSeries (DirectoryObject dirObj) throws PersistException
    {
        while (dirObj.getParent() != null)
            dirObj = (DirectoryObject) dirObj.getParent();
        if (!this.isSeriesDirectory(dirObj))
        {
            throw new PersistException(PersistException.E_LOAD_SERIES,
                dirObj, null, "Not a series directory");
        }
        dirObj.clear();
        PersistFactory.getPersist(m_settings).load(dirObj);
    } // loadSeries

    /**
     * Saves the series where <tt>dirObj</tt> is part of.
     * @param dirObj - The directory object
     * @param bForce - <tt>True</tt> to force saving even if the series has not
     * changed
     * @throws PersistException on save error
     */
    public void saveSeries
        ( DirectoryObject dirObj
        , boolean         bForce
        ) throws PersistException
    {
        while (dirObj.getParent() != null)
            dirObj = (DirectoryObject) dirObj.getParent();
        if (!this.isSeriesDirectory(dirObj))
        {
            throw new PersistException(PersistException.E_SAVE_SERIES,
                dirObj, null, "Not a series directory");
        }
        PersistFactory.getPersist(m_settings).save(dirObj, bForce);
    } // saveSeries

    /**
     * Saves the series where <tt>dirObj</tt> is part of for the Web.
     * @param dirObj - The directory object
     * @throws PersistException on save error
     */
    public void saveSeriesForWeb (DirectoryObject dirObj)
        throws PersistException
    {
        while (dirObj.getParent() != null)
            dirObj = (DirectoryObject) dirObj.getParent();
        if (!this.isSeriesDirectory(dirObj))
        {
            throw new PersistException(PersistException.E_SAVE_SERIES,
                dirObj, null, "Not a series directory");
        }
        PersistFactory.getPersist(m_settings).saveForWeb(dirObj);
    } // saveSeriesForWeb
} // SeriesContainer
