/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: AbstractDirPersist
    Created: 28.01.2006 (17:08:32)
        $Id: AbstractDirPersist.java 159 2009-05-19 19:40:47Z dirk $
  $Revision: 159 $
      $Date: 2009-05-19 21:40:47 +0200 (Di, 19 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.dir.persist;

import java.io.File;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;

import com.dgrossmann.photo.dir.AbstractFSObject;
import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.settings.Settings;

/**
 * Abstract base class for series and directory persister classes. Provides the
 * common functionality.
 * @author Dirk Grossmann
 */
public abstract class AbstractDirPersist implements IDirPersist
{
    private Settings m_settings;

    /**
     * Creates a new <tt>AbstractDirPersist</tt> instance.
     * @param settings - The Settings object
     */
    public AbstractDirPersist (Settings settings)
    {
        super();
        m_settings = settings;
    } // AbstractDirPersist

    /**
     * Gets the settings object.
     * @return The settings
     */
    protected Settings getSettings ()
    {
        return m_settings;
    } // getSettings

    /**
     * Protected helper template method that returns the series metadata store
     * name.
     * @param seriesDirObj - Series root directory object
     * @param bAbsolute - <tt>True</tt> to get the absolute path
     * @return Full meta data store name as string
     */
    protected abstract String getMetaDataStoreName
        ( DirectoryObject seriesDirObj
        , boolean         bAbsolute
        );

    /**
     * Protected helper template method for <tt>load</tt>. It reads the meta
     * data store and fills the directory with reference objects.
     * @param seriesDirObj - Series directory object
     * @param metaDataFileName - Meta data store name
     * @throws PersistException on error
     */
    protected abstract void readMetaDataStore
        ( DirectoryObject seriesDirObj
        , String          metaDataFileName
        ) throws PersistException;

    /**
     * Loads the information for the series directory <tt>seriesDirObj</tt>
     * from the file system and the meta data repository.
     * @param seriesDirObj - Series root directory obbject. It should be set up
     * with the absolute path of the directory, but is otherwise empty.
     * @throws PersistException on load error
     * @see com.dgrossmann.photo.dir.persist.IDirPersist#load(com.dgrossmann.photo.dir.DirectoryObject)
     */
    public void load (DirectoryObject seriesDirObj) throws PersistException
    {
        File dirFile;

        dirFile = new File(seriesDirObj.getFullPath());
        if (!dirFile.isDirectory())
        {
            PersistException pe = new PersistException
                (PersistException.E_LOAD_SERIES,
                 seriesDirObj, "", "Series directory is not a directory");
            pe.log();
            throw pe;
        }
        seriesDirObj.clear();
        // Check for meta data file and read it.
        this.readMetaDataStore(seriesDirObj,
            this.getMetaDataStoreName(seriesDirObj, true));
        // Read the files and sub directories and merge them with the meta data
        // read from the store.
        this.scanDirectory(seriesDirObj);
    } // load

    /**
     * Private helper method to scan a directory tree.
     * @param dirObj - Represents the root of the directory tree to be scanned
     */
    private void scanDirectory (DirectoryObject dirObj)
    {
        DirectoryObject             subDirObj;
        FileObject                  fileObj;
        File                        f;
        String[]                    contents;
        ArrayList<File>             dirObjects, fileObjects;
        Iterator<File>              fsFileIter;
        ArrayList<AbstractFSObject> emptyRefs;
        Iterator<AbstractFSObject>  refIter;
        Iterator<DirectoryObject>   dirIter;
        Iterator<FileObject>        fileIter;
        String                      name;
        int                         i;

        // Read the directory contents.
        dirObjects = new ArrayList<File>();
        fileObjects = new ArrayList<File>();
        contents = (new File(dirObj.getFullPath())).list();
        if (contents == null)
            return;
        for (i = 0; i < contents.length; i++)
        {
            if (contents[i].startsWith(".") || contents[i].startsWith("_") ||
                contents[i].startsWith("#") || contents[i].startsWith("~") ||
                contents[i].equalsIgnoreCase("thumbs.db"))
            {
                continue;
            }
            f = new File(dirObj.getFullPath() + File.separator + contents[i]);
            if (f.isDirectory())
                dirObjects.add(f);
            else
                fileObjects.add(f);
        }
        Collections.sort(dirObjects, new FSObjectComparator());
        Collections.sort(fileObjects, new FSObjectComparator());
        // Process the sub directories.
        fsFileIter = dirObjects.iterator();
        while (fsFileIter.hasNext())
        {
            name = fsFileIter.next().getName();
            subDirObj = this.incorporateSubDir(dirObj, name);
            if (subDirObj == null)
            {
                subDirObj = new DirectoryObject(name, dirObj);
                subDirObj.setReference(false);
                dirObj.addChild(subDirObj, -1);
            }
            this.scanDirectory(subDirObj);
        }
        // Process the files.
        fsFileIter = fileObjects.iterator();
        while (fsFileIter.hasNext())
        {
            name = fsFileIter.next().getName();
            if (!this.incorporateFile(dirObj, name))
            {
                fileObj = new FileObject(name, dirObj);
                fileObj.setPropertiesFromImageMetadata();
                fileObj.setReference(false);
                dirObj.addChild(fileObj, -1);
            }
        }
        // Remove remaining empty file references.
        emptyRefs = new ArrayList<AbstractFSObject>(dirObj.getFileCount());
        fileIter = dirObj.getFileIterator();
        while (fileIter.hasNext())
        {
            fileObj = fileIter.next();
            if (fileObj.isEmpty())
                emptyRefs.add(fileObj);
        }
        refIter = emptyRefs.iterator();
        while (refIter.hasNext())
            dirObj.removeFile((FileObject) refIter.next());
        // Remove remaining empty sub directory references.
        emptyRefs.clear();
        dirIter = dirObj.getSubDirIterator();
        while (dirIter.hasNext())
        {
            subDirObj = dirIter.next();
            if (subDirObj.isEmpty())
                emptyRefs.add(subDirObj);
        }
        refIter = emptyRefs.iterator();
        while (refIter.hasNext())
            dirObj.removeSubDir((DirectoryObject) refIter.next());
    } // scanDirectory

    /**
     * Private helper method for <tt>scanDirectory</tt> that tries to
     * incorporate a sub directory into an empty directory object that has been
     * created from a directory file.
     * @param dirObj - Parent directory object
     * @param fileName - Name of the sub directory
     * @return Returns the sub directory object that has been incorporated or
     * <tt>null</tt> if it has not been incorporated
     */
    private DirectoryObject incorporateSubDir
        ( DirectoryObject dirObj
        , String          fileName
        )
    {
        Iterator<DirectoryObject> iter;
        DirectoryObject           subDirObj;
        String                    fnpart;

        iter = dirObj.getSubDirIterator();
        while (iter.hasNext())
        {
            subDirObj = iter.next();
            if (subDirObj == null)
                continue;
            fnpart = subDirObj.getFileNamePart();
            if (fnpart.length() == 0)
                continue;
            // To match, the sub directory name must equal the file name.
            if (fnpart.equalsIgnoreCase(fileName))
            {
                subDirObj.setFileName(fileName, false);
                subDirObj.setFileNamePart("");
                subDirObj.setReference(false);
                return subDirObj;
            }
        }
        return null;
    } // incorporateSubDir

    /**
     * Private helper method for <tt>scanDirectory</tt> that tries to
     * incorporate a file into an empty file object that has been created from a
     * directory file.
     * @param dirObj - Parent directory object
     * @param fileName - File name
     * @return Returns whether the file has been incorporated
     */
    private boolean incorporateFile (DirectoryObject dirObj, String fileName)
    {
        Iterator<FileObject> fileIter;
        FileObject           fileObj;
        String               name, fnpart;
        int                  partLength;
        boolean              bMatch;

        name = fileName.toLowerCase();
        fileIter = dirObj.getFileIterator();
        while (fileIter.hasNext())
        {
            fileObj = fileIter.next();
            if (fileObj == null)
                continue;
            fnpart = fileObj.getFileNamePart();
            if (fnpart.length() == 0)
                continue;
            fnpart = fnpart.toLowerCase();
            // To match, the file name must start with the file name part;
            // after this, the string must end or the next character must be
            // a period.
            if (fnpart.endsWith("."))
                fnpart = fnpart.substring(0, fnpart.length()-1);
            bMatch = (name.indexOf(fnpart) >= 0);
            if (bMatch)
            {
                partLength = fnpart.length();
                bMatch = (partLength == name.length() ||
                    name.charAt(partLength) == '.');
            }
            if (bMatch)
            {
                fileObj.setFileName(fileName, false);
                fileObj.setFileNamePart("");
                fileObj.setReference(false);
                if (fileObj.getTitle(false).length() == 0)
                	fileObj.setPropertiesFromImageMetadata();
                return true;
            }
        }
        return false;
    } // incorporateFile

    /**
     * Helper template method that saves a series into the meta data store.
     * @param seriesDirObj - The series to save
     * @param metadataStoreName - The file name of the meta data store
     * @param bForWeb - <tt>True</tt> to save for the Web export directory;
     * <tt>false</tt> to save for the original series
     * @throws PersistException on save error
     */
    protected abstract void internalSave
        ( DirectoryObject seriesDirObj
        , String          metadataStoreName
        , boolean         bForWeb
        ) throws PersistException;

    /**
     * Saves the information for the series directory <tt>seriesDirObj</tt> and
     * its sub objects into the meta data repository.
     * @param seriesDirObj - Series root directory object to be saved
     * @param bForce - <tt>True</tt> to force saving
     * @throws PersistException on save error
     * @see com.dgrossmann.photo.dir.persist.IDirPersist#save(com.dgrossmann.photo.dir.DirectoryObject, boolean)
     */
    public void save
        ( DirectoryObject seriesDirObj
        , boolean         bForce
        ) throws PersistException
    {
        if (bForce || seriesDirObj.isChanged())
        {
	        this.internalSave(seriesDirObj,
	            this.getMetaDataStoreName(seriesDirObj, true), false);
        }
    } // save

    /**
     * Saves the information for the series directory <tt>seriesDirObj</tt> and
     * its sub objects into the meta data repository for the Web.
     * @param seriesDirObj - Series root directory object to be saved
     * @throws PersistException on save error
     * @see com.dgrossmann.photo.dir.persist.IDirPersist#saveForWeb(com.dgrossmann.photo.dir.DirectoryObject)
     */
    public void saveForWeb (DirectoryObject seriesDirObj)
        throws PersistException
    {
        File outDir = new File(this.getSettings().get
            (Settings.EXPORT_DIRECTORY));
        if (!outDir.isDirectory())
        {
            throw new PersistException(PersistException.E_SAVE_SERIES,
                seriesDirObj, null, "No Web export directory");
        }
        // Create the directory on demand.
        File dir = new File(outDir, seriesDirObj.getFileName().toLowerCase());
        if (!dir.isDirectory())
            dir.mkdir();
        // Get the meta data file and save their series information.
        String metaDataFile = outDir + File.separator
            + this.getMetaDataStoreName(seriesDirObj, false);
        this.internalSave(seriesDirObj, metaDataFile, true);
    } // saveForWeb
} // AbstractDirPersist

/**
 * Helper class for the directory persisters to sort files and directories
 * according to their modification date. It tries to sort such that the oldest
 * objects come first.
 * @author Dirk Grossmann
 */
class FSObjectComparator implements Comparator<Object>
{
    /**
     * Compares its two arguments for order. An object is less than the other
     * if it is older.
     * @param obj1 - First object to compare
     * @param obj2 - Second object
     * @return -1 if obj1 &lt; obj2, 0 if obj1 == obj2, 1 if obj1 &gt; obj2
     */
    public int compare (Object obj1, Object obj2)
    {
        Calendar mod1, mod2;
        long     ts1, ts2;

        if (obj1 == null || obj2 == null)
        {
            if (obj1 == null)
                return (obj2 == null) ? 0 : -1;
            return 1;
        }
        if (obj1 instanceof File && obj2 instanceof File)
        {
            ts1 = ((File) obj1).lastModified();
            ts2 = ((File) obj2).lastModified();
            if (ts1 < ts2)
                return -1;
            if (ts1 > ts2)
                return 1;
            return 0;
        }
        if (obj1 instanceof AbstractFSObject &&
            obj2 instanceof AbstractFSObject)
        {
            mod1 = ((AbstractFSObject) obj1).getModDateTime();
            mod2 = ((AbstractFSObject) obj2).getModDateTime();
            if (mod1 != null || mod2 != null)
            {
                if (mod1 == null)
                    return 1;
                if (mod2 == null)
                    return -1;
                ts1 = mod1.getTimeInMillis();
                ts2 = mod2.getTimeInMillis();
                if (ts1 < ts2)
                    return -1;
                if (ts1 > ts2)
                    return 1;
                return 0;
            }
        }
        return obj1.toString().compareTo(obj2.toString());
    } // compare
} // FSObjectComparator
