/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: DirPersistFile
    Created: 2 January, 2003
        $Id: DirPersistFile.java 158 2009-05-06 19:49:13Z dirk $
  $Revision: 158 $
      $Date: 2009-05-06 21:49:13 +0200 (Mi, 06 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.dir.persist;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.PrintWriter;
import java.util.Iterator;

import com.dgrossmann.photo.AppInfo;
import com.dgrossmann.photo.dir.AbstractFSObject;
import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.settings.Settings;

/** Helper class to represent a named value. */
class NamedValue
{
    public String name;
    public String value;
    public NamedValue ()
    {
        name = value = "";
    } // NamedValue
    public NamedValue (String n, String v)
    {
        name = n;
        value = v;
    } // NamedValue
} // NamedValue

/**
 * Instances of this class are used to persist the meta data of a series
 * directory into ASCII files.
 */
public class DirPersistFile extends AbstractDirPersist
{
    private static final String FOR_WEB         = "web";
    private static final String SERIES          = "series";
    private static final String GROUP           = "group";
    private static final String FILE            = "file";
    private static final String TO_EXPORT       = "to-export";
    private static final String EXPORT_QUALITY  = "export-quality";
    private static final String END             = "end";
    private static final String END_FILE        = "end-file";
    private static final String END_DESCRIPTION = "end-description";

    /**
     * Protected constructor of a <tt>DirPersistFile</tt> instance.
     * @param settings - The Settings object
     */
    public DirPersistFile (Settings settings)
    {
        super(settings);
    } // DirPersistFile

    /**
     * Helper that returns the series metadata file name.
     * @param seriesDirObj - Series root directory object
     * @param bAbsolute - <tt>True</tt> to get the absolute paths
     * @return Full path of the metadata file name as string
     * @see com.dgrossmann.photo.dir.persist.AbstractDirPersist#getMetaDataStoreName(com.dgrossmann.photo.dir.DirectoryObject, boolean)
     */
    @Override
	protected String getMetaDataStoreName
        ( DirectoryObject seriesDirObj
        , boolean         bAbsolute
        )
    {
        String name = seriesDirObj.getFileName().toLowerCase();
        name = '_' + name.replace(' ', '-') + "_metadata.txt";
        if (!bAbsolute)
        {
            return seriesDirObj.getFileName().toLowerCase()
                + File.separator + name;
        }
        return seriesDirObj.getFullPath() + File.separator + name;
    } // getMetaDataStoreName

    /**
     * @see com.dgrossmann.photo.dir.persist.IDirPersist#canLoad(com.dgrossmann.photo.dir.DirectoryObject)
     */
    public boolean canLoad (DirectoryObject seriesDirObj)
    {
        String fileName = this.getMetaDataStoreName(seriesDirObj, true);
        return (new File(fileName)).exists();
    } // canLoad

    /**
     * Protected helper method for <tt>load</tt>. It reads the meta data file
     * and fills the directory with reference objects.
     * @param seriesDirObj - Series directory object
     * @param metaDataFileName - Meta data file name
     * @throws PersistException on error
     */
    @Override
	protected void readMetaDataStore
        ( DirectoryObject seriesDirObj
        , String          metaDataFileName
        ) throws PersistException
    {
        File           metaDataFile;
        BufferedReader in;
        String         line;
        NamedValue     val;

        metaDataFile = new File(metaDataFileName);
        if (!metaDataFile.exists())
            return;
        in = null;
        try
        {
            in = new BufferedReader(new FileReader(metaDataFile));
            line = "";
            val = new NamedValue();
            // Read the series data.
            line = this.readValue(val, in, line);
            if (val.name.equals(FOR_WEB))
            {
                PersistException pe = new PersistException(
                    PersistException.E_LOAD_SERIES,
                    seriesDirObj, metaDataFile.getAbsolutePath(),
                    "This is a metadata file for a Web directory and "
                    + "cannot be loaded as series directory");
                pe.log();
                throw pe;
            }
            if (!val.name.equals(SERIES))
            {
                PersistException pe = new PersistException(
                    PersistException.E_LOAD_SERIES,
                    seriesDirObj, metaDataFile.getAbsolutePath(),
                    "@series expected near line:\n" + line);
                pe.log();
                throw pe;
            }
            if (line != null)
                line = this.readDescription(seriesDirObj, val.value, in, line);
            else
            {
                PersistException pe = new PersistException(
                    PersistException.E_LOAD_SERIES,
                    seriesDirObj, metaDataFile.getAbsolutePath(),
                    "Unexpected end-of-file after series begin");
                pe.log();
                throw pe;
            }
            if (line != null)
            {
                // Read the file/group block (possibly recursively.
                line = this.readGroup(seriesDirObj, in, line);
            }
            else
            {
                PersistException pe = new PersistException(
                    PersistException.E_LOAD_SERIES,
                    seriesDirObj, metaDataFile.getAbsolutePath(),
                    "Unexpected end-of-file after series description");
                pe.log();
                throw pe;
            }
        }
        catch (Exception e)
        {
            PersistException pe = new PersistException
                (PersistException.E_LOAD_SERIES,
                 seriesDirObj, metaDataFile.getAbsolutePath(), e);
            pe.log();
            throw pe;
        }
        finally
        {
            try
            {
                if (in != null)
                    in.close();
            }
            catch (Exception ignored)
            {
            }
        }
    } // readMetaDataStore

    /**
     * Reads the meta data for a series or group.
     * @param parentDirObj - Parent directory object
     * @param in - Buffered reader representing the meta data file
     * @param line - Last line read by a previous <tt>readValue</tt> method call
     * @return String representing the last read line to be passed to the next
     * <tt>readValue</tt> method call or <tt>null</tt> for end-of-file
     * @throws Exception on error
     */
    private String readGroup
        ( DirectoryObject parentDirObj
        , BufferedReader  in
        , String          line
        ) throws Exception
    {
        DirectoryObject dirObj;
        FileObject      fileObj;
        NamedValue      val;

        val = new NamedValue();
        // A block is a sequence of files and blocks, terminated by "@end".
        while (line != null)
        {
            line = this.readValue(val, in, line);
            if (val.name.equals(END))
                return line;
            if (val.name.equals(FILE))
            {
                // This is a file.
                fileObj = new FileObject("", parentDirObj);
                parentDirObj.addChild(fileObj, -1);
                line = this.readDescription(fileObj, val.value, in, line);
                continue;
            }
            if (val.name.equals(GROUP))
            {
                // This is a directory.
                dirObj = new DirectoryObject("", parentDirObj);
                parentDirObj.addChild(dirObj, -1);
                line = this.readDescription(dirObj, val.value, in, line);
                // Read the directory contents.
                line = this.readGroup(dirObj, in, line);
                continue;
            }
            throw new Exception("@file, @group or @end expected at (@"
                + val.name + ' ' + val.value + ") near line: " + line);
        }
        return null;
    } // readGroup

    /**
     * Reads the description for a file system entry from the meta data file
     * and fills it.
     * @param fsObj - Object filled with the description
     * @param fnPart - File name part from the meta data file
     * @param in - Buffered reader representing the meta data file
     * @param line - Last line read by a previous <tt>readValue</tt> method call
     * @return String representing the last read line to be passed to the next
     * <tt>readValue</tt> method call or <tt>null</tt> for end-of-file
     * @throws Exception on error
     */
    private String readDescription
        ( AbstractFSObject fsObj
        , String           fnPart
        , BufferedReader   in
        , String           line
        ) throws Exception
    {
        NamedValue val;
        int        quality;

        val = new NamedValue();
        fsObj.setFileNamePart(fnPart);
        while (line != null)
        {
            line = this.readValue(val, in, line);
            if (val.name.equals(END_DESCRIPTION) ||
                val.name.equals(END_FILE))
            {
                return line;
            }
            else if (val.name.equals(TO_EXPORT))
            {
                char ch = (val.value.length() > 0) ? val.value.charAt(0) : 'f';
                fsObj.setToExport(ch == 't' || ch == 'y' || ch == '1');
            }
            else if (val.name.equals(EXPORT_QUALITY))
            {
                try
                {
                    quality = Integer.parseInt(val.value);
                    fsObj.setConversionQuality(quality);
                }
                catch (Exception ignored)
                {
                }
            }
            else
                fsObj.set(val.name, val.value);
        }
        return null;
    } // readDescription

    /**
     * Private helper for load: reads a "@named value" from the meta data file.
     * @param val - Named value object to be filled
     * @param in - Buffered reader representing the meta data file
     * @param line - Last line read by a previous method call
     * @return String representing the last read line to be passed to the next
     * method call or <tt>null</tt> for end-of-file
     * @throws Exception on error
     */
    private String readValue
        ( NamedValue     val
        , BufferedReader in
        , String         line
        ) throws Exception
    {
        val.name = val.value = "";
        // Go to the next '@' line.
        while (line.length() == 0 || line.charAt(0) != '@')
        {
            line = in.readLine();
            if (line == null)
                return null;
            line = line.trim();
            if (line.length() == 0 || line.charAt(0) == '#')
                continue;
            if (line.charAt(0) != '@')
                throw new Exception("Unexpected line: " + line);
        }
        // Split the "@name val ..." string.
        line = line.substring(1);
        int i = line.indexOf(' ');
        if (i >= 0)
        {
            val.name = line.substring(0, i).toLowerCase();
            val.value = line.substring(i).trim();
        }
        else
            val.name = line.trim().toLowerCase();
        // Test for an end token without a value.
        if (val.name.startsWith(END))
        {
            val.value = "";
            // Here, we do not proceed reading and indicate that the current
            // line is consumed.
            return "";
        }
        // The value can encompass the following lines until a "@..." line is
        // encountered.
        while ((line = in.readLine()) != null)
        {
            if (line.trim().startsWith("@"))
                return line.trim();
            val.value += '\n' + line;
        }
        // This should never happen.
        return null;
    } // readValue

    /**
     * @see com.dgrossmann.photo.dir.persist.AbstractDirPersist#internalSave(com.dgrossmann.photo.dir.DirectoryObject, java.lang.String, boolean)
     */
    @Override
	protected void internalSave
        ( DirectoryObject seriesDirObj
        , String          metadataStoreName
        , boolean         bForWeb
        ) throws PersistException
    {
        File        f   = new File(metadataStoreName);
        PrintWriter out = null;

        if (seriesDirObj.isError())
        {
            throw new PersistException(PersistException.E_SAVE_SERIES,
                seriesDirObj, null, "Cannot be saved because loading failed");
        }
        try
        {
            out = new PrintWriter(new FileWriter(f));
            // Write series information.
            out.println("# " + AppInfo.APP_NAME + " (Version "
                + AppInfo.getVersionString()
                + ") photo series metadata file.");
            if (bForWeb)
            {
                out.println("# Metadata exported for the Web. Do not load "
                    + "as regular series metadata.");
                out.println();
                this.writeValue(FOR_WEB, "", out, 0);
            }
            out.println();
            String name = seriesDirObj.getFileName();
            if (bForWeb)
                name = name.toLowerCase();
            out.println("@series " + name);
            this.writeDescription(seriesDirObj, bForWeb, out, 1);
            // Write the series contents.
            this.writeGroupBody(seriesDirObj, bForWeb, out, 1);
            out.println("@end");
        }
        catch (Exception exc)
        {
            PersistException pe = new PersistException
                (PersistException.E_SAVE_SERIES,
                 seriesDirObj, f.getAbsolutePath(), exc);
            pe.log();
            throw pe;
        }
        finally
        {
            try
            {
                if (out != null)
                    out.close();
            }
            catch (Exception ignored)
            {
            }
        }
    } // internalSave

    /**
     * Private helper for <tt>internalSave</tt> to write the group body
     * (without directory name, description and end marker).
     * @param dirObj - Directory object whose contained files and subdirectories
     * should be saved
     * @param bForWeb - <tt>True</tt> to write for web export
     * @param out - Print writer representing the output meta data file
     * @param indent - Indent for beautifying the output file
     * @throws Exception on error
     */
    private void writeGroupBody
        ( DirectoryObject dirObj
        , boolean         bForWeb
        , PrintWriter     out
        , int             indent
        ) throws Exception
    {
        Iterator<DirectoryObject> subdirIter;
        Iterator<FileObject>      fileIter;
        DirectoryObject           subDir;
        FileObject                fileObj;
        String                    name;

        subdirIter = dirObj.getSubDirIterator();
        while (subdirIter.hasNext())
        {
            subDir = subdirIter.next();
            if (bForWeb && !subDir.isToExport())
                continue;
            out.println();
            name = subDir.getFileName();
            if (name.length() == 0)
                name = subDir.getFileNamePart();
            if (bForWeb)
                name = name.toLowerCase();
            this.writeValue(GROUP, name, out, indent);
            this.writeDescription(subDir, bForWeb, out, indent+1);
            this.writeGroupBody(subDir, bForWeb, out, indent+1);
            this.writeValue(END, "", out, indent);
        }
        fileIter = dirObj.getFileIterator();
        while (fileIter.hasNext())
        {
            fileObj = fileIter.next();
            if (bForWeb && !fileObj.isToExport())
                continue;
            out.println();
            if (fileObj.isReference())
                name = fileObj.getFileNamePart();
            else if (bForWeb)
            {
                if (fileObj.getFileType() != FileObject.TYPE_IMAGE_PREVIEW)
                    name = fileObj.getFileName();
                else
                {
                    // Change for other JPEG image types.
                    name = fileObj.getBaseFileName() + ".jpg";
                }
                name = name.toLowerCase();
            }
            else
                name = fileObj.getFileName();
            if (name.length() == 0)
                name = fileObj.getFileNamePart();
            this.writeValue(FILE, name, out, indent);
            this.writeDescription(fileObj, bForWeb, out, indent);
        }
    } // writeGroupBody

    /**
     * Private helper for <tt>internalSave</tt> to write the properties of a
     * file system object.
     * @param fsObj - File system object
     * @param bForWeb - <tt>True</tt> to write for web export
     * @param out - Print writer representing the output meta data file
     * @param indent - Indent for beautifying the output file
     * @throws Exception on error
     */
    private void writeDescription
        ( AbstractFSObject fsObj
        , boolean          bForWeb
        , PrintWriter      out
        , int              indent
        ) throws Exception
    {
        String[] propNames;
        String   val;
        int      i;

        // Write the file system object properties.
        propNames = fsObj.getUsedPropertyNames();
        for (i = 0; i < propNames.length; i++)
        {
            val = fsObj.getTransformed(propNames[i], true);
            if (val.length() > 0)
                this.writeValue(propNames[i], val, out, indent);
        }
        // Write the fixed properties.
        if (bForWeb)
        {
            // Write the guessed title if empty.
            val = fsObj.getTitle(false);
            if (val.length() == 0)
            {
                this.writeValue(AbstractFSObject.TITLE, fsObj.getTitle(true),
                    out, indent);
            }
        }
        else
        {
            // Write the export settings only when not exporting.
            if (fsObj.isToExport())
            {
                this.writeValue(TO_EXPORT, "yes", out, indent);
                if (fsObj.getConversionQuality() > 0)
                {
                    this.writeValue(EXPORT_QUALITY,
                        Integer.toString(fsObj.getConversionQuality()),
                        out, indent);
                }
            }
            else
                this.writeValue(TO_EXPORT, "no", out, indent);
        }
        // Write the end.
        if (fsObj instanceof FileObject)
            this.writeValue(END_FILE, "", out, indent);
        else
            this.writeValue(END_DESCRIPTION, "", out, indent);
    } // writeDescription

    /**
     * Private helper for <tt>internalSave</tt> to write a name/value pair.
     * @param name - Property name
     * @param value - Property value
     * @param out - Print writer representing the output meta data file
     * @param indent - Indent for beautifying the output file
     * @throws Exception on error
     */
    private void writeValue
        ( String      name
        , String      value
        , PrintWriter out
        , int         indent
        ) throws Exception
    {
        int i, iOld;

        for (i = 0; i < indent; i++)
            out.print(' ');
        out.print('@');
        out.print(name);
        if (value.length() > 0)
        {
            out.print(' ');
            // Print the value as possible multi-line string.
            iOld = 0;
            while ((i = value.indexOf("\n", iOld)) >= 0)
            {
                out.println(value.substring(iOld, i));
                iOld = i + 1;
            }
            out.println(value.substring(iOld));
        }
        else
            out.println();
    } // writeValue
} // DirPersistFile
