/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: DirPersistXmlFile
    Created: 28.01.2006 (17:28:41)
        $Id: DirPersistXmlFile.java 159 2009-05-19 19:40:47Z dirk $
  $Revision: 159 $
      $Date: 2009-05-19 21:40:47 +0200 (Di, 19 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.dir.persist;

import java.io.File;
import java.io.FileOutputStream;
import java.io.PrintWriter;
import java.text.DateFormat;
import java.util.Date;
import java.util.Iterator;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Attr;
import org.w3c.dom.Comment;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.dgrossmann.photo.AppInfo;
import com.dgrossmann.photo.dir.AbstractFSObject;
import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.settings.Settings;

/**
 * Instances of this class are used to persist the meta data of a series
 * directory into XML files.
 * @author Dirk Grossmann
 */
public class DirPersistXmlFile extends AbstractDirPersist
{
    private static final String DEFAULT_NAMESPACE =
        "http://www.dgrossmann.com/photo/metadata";

    // XML element names.
    private static final String ELEM_SERIES          = "series";
    private static final String ELEM_GROUP           = "group";
    private static final String ELEM_CONTENTS        = "contents";
    private static final String ELEM_SEPARATOR       = "separator";
    private static final String ELEM_FILE            = "file";
    private static final String ELEM_TITLE           = "title";
    private static final String ELEM_SUBTITLE        = "subtitle";
    private static final String ELEM_LOCATION        = "location";
    private static final String ELEM_DESCRIPTION     = "description";
    private static final String ELEM_REMARK          = "remark";
    private static final String ELEM_PROPERTY        = "property";
    // XML attribute names.
    private static final String ATTR_NAME            = "name";
    private static final String ATTR_REFERENCE       = "is-reference";
    private static final String ATTR_HREF            = "href";
    private static final String ATTR_BEGIN_DATE      = "begin-date";
    private static final String ATTR_END_DATE        = "end-date";
    private static final String ATTR_EXPORT          = "export";
    private static final String ATTR_EXPORT_QUALITY  = "quality";
    private static final String ATTR_WEB_EXPORTED    = "web-exported";

    private String m_saveXmlEncoding;
    private String m_exportXmlEncoding;

    /**
     * Creates a new <tt>DirPersistXmlFile</tt> instance.
     * @param settings
     */
    public DirPersistXmlFile (Settings settings)
    {
        super(settings);
        m_saveXmlEncoding = m_exportXmlEncoding =
            Settings.METADEFAULT_XML_ENCODING;
    } // DirPersistXmlFile

    /**
     * Sets the XML encoding strings.
     * @param saveXmlEncoding - For saving
     * @param exportXmlEncoding - For Web-exporting
     */
    public void setEncodings (String saveXmlEncoding, String exportXmlEncoding)
    {
        m_saveXmlEncoding = saveXmlEncoding;
        m_exportXmlEncoding = exportXmlEncoding;
    } // setEncodings

    /**
     * @see com.dgrossmann.photo.dir.persist.AbstractDirPersist#getMetaDataStoreName(com.dgrossmann.photo.dir.DirectoryObject, boolean)
     */
    @Override
	protected String getMetaDataStoreName
        ( DirectoryObject seriesDirObj
        , boolean         bAbsolute
        )
    {
        String name = seriesDirObj.getFileName().toLowerCase();
        name = '_' + name.replace(' ', '-') + "-metadata.xml";
        if (!bAbsolute)
        {
            return seriesDirObj.getFileName().toLowerCase()
                + File.separator + name;
        }
        return seriesDirObj.getFullPath() + File.separator + name;
    } // getMetaDataStoreName

    /**
     * @see com.dgrossmann.photo.dir.persist.IDirPersist#canLoad(com.dgrossmann.photo.dir.DirectoryObject)
     */
    public boolean canLoad (DirectoryObject seriesDirObj)
    {
        String fileName = this.getMetaDataStoreName(seriesDirObj, true);
        return (new File(fileName)).exists();
    } // canLoad

    /**
     * @see com.dgrossmann.photo.dir.persist.AbstractDirPersist#readMetaDataStore(com.dgrossmann.photo.dir.DirectoryObject, java.lang.String)
     */
    @Override
	protected void readMetaDataStore
        ( DirectoryObject seriesDirObj
        , String          metaDataFileName
        ) throws PersistException
    {
        File                   metaDataFile;
        DocumentBuilderFactory fac;
        DocumentBuilder        builder;
        Document               doc;
        Element                rootElem;

        metaDataFile = new File(metaDataFileName);
        try
        {
            fac = DocumentBuilderFactory.newInstance();
            fac.setValidating(false);
            builder = fac.newDocumentBuilder();
            doc = builder.parse(metaDataFile);
            // The root element is the series.
            rootElem = doc.getDocumentElement();
            if (!rootElem.getTagName().equals(ELEM_SERIES))
            {
                PersistException pe = new PersistException(
                    PersistException.E_LOAD_SERIES,
                    seriesDirObj, metaDataFile.getAbsolutePath(),
                    "This file is not a JPhoto-Explorer series meta data file "
                    + "as it does not have the <series> root element");
                pe.log();
                throw pe;
            }
            if (rootElem.hasAttribute(ATTR_WEB_EXPORTED))
            {
                PersistException pe = new PersistException(
                    PersistException.E_LOAD_SERIES,
                    seriesDirObj, metaDataFile.getAbsolutePath(),
                    "This is a metadata file for a Web directory and "
                    + "cannot be loaded as series directory");
                pe.log();
                throw pe;
            }
            // Read the series description and the groups.
            this.readDescription(seriesDirObj, rootElem);
            this.readGroup(seriesDirObj, rootElem);
        }
        catch (Exception exc)
        {
            PersistException pe = new PersistException(
                PersistException.E_LOAD_SERIES, seriesDirObj,
                metaDataFile.getAbsolutePath(), exc);
            pe.log();
            throw pe;
        }
    } // readMetaDataStore

    /**
     * Private helper method to read the meta data for a series or group.
     * @param parentDirObj - Parent directory object
     * @param dirElement - The XML source element representing this directory
     * object
     */
    private void readGroup
        ( DirectoryObject parentDirObj
        , Element         dirElement
        )
    {
        DirectoryObject dirObj;
        FileObject      fileObj;
        NodeList        children;
        Node            n;
        Element         contentsElement, elem;
        String          name;
        int             i;

        // Get the contents element.
        contentsElement = null;
        children = dirElement.getChildNodes();
        for (i = 0; i < children.getLength(); i++)
        {
            n = children.item(i);
            if (!(n instanceof Element))
                continue;
            elem = (Element) n;
            if (elem.getTagName().equals(ELEM_CONTENTS))
            {
                contentsElement = elem;
                break;
            }
        }
        if (contentsElement == null)
            return;
        // Get the sub-elements of the contents.
        children = contentsElement.getChildNodes();
        for (i = 0; i < children.getLength(); i++)
        {
            n = children.item(i);
            if (!(n instanceof Element))
                continue;
            elem = (Element) n;
            name = elem.getTagName();
            // Create the sub-fs-objects by tag name: group, file, separator.
            if (name.equals(ELEM_GROUP))
            {
                dirObj = new DirectoryObject("", parentDirObj);
                parentDirObj.addChild(dirObj, -1);
                this.readDescription(dirObj, elem);
                // Read the directory contents.
                this.readGroup(dirObj, elem);
            }
            else if (name.equals(ELEM_FILE) || name.equals(ELEM_SEPARATOR))
            {
                fileObj = new FileObject("", parentDirObj);
                parentDirObj.addChild(fileObj, -1);
                this.readDescription(fileObj, elem);
                if (name.equals(ELEM_SEPARATOR))
                    fileObj.makeSeparator();
            }
        }
    } // readGroup

    /**
     * Private helper method to read the meta data for the description of a file
     * system object.
     * @param fsObj - The file system object to fill with the description
     * @param fsElement - The XML source element representing this file system
     * object
     */
    private void readDescription
        ( AbstractFSObject fsObj
        , Element          fsElement
        )
    {
        Node   subNode;
        Attr   attrib;
        String name, value;
        int    quality;
        char   ch;

        // Get the name.
        attrib = fsElement.getAttributeNode(ATTR_NAME);
        if (attrib != null)
            fsObj.setFileNamePart(attrib.getValue());
        // Get the begin and end date.
        attrib = fsElement.getAttributeNode(ATTR_BEGIN_DATE);
        if (attrib != null)
            fsObj.set(AbstractFSObject.DATE_BEGIN, attrib.getValue());
        attrib = fsElement.getAttributeNode(ATTR_END_DATE);
        if (attrib != null)
            fsObj.set(AbstractFSObject.DATE_END, attrib.getValue());
        // Get whether it is a reference.
        if (fsElement.hasAttribute(ATTR_REFERENCE))
        {
            fsObj.setReference(true);
            attrib = fsElement.getAttributeNode(ATTR_HREF);
            if (attrib != null)
                fsObj.set(AbstractFSObject.HREF, attrib.getValue());
        }
        // Get the export parameters.
        attrib = fsElement.getAttributeNode(ATTR_EXPORT);
        if (attrib != null)
        {
            // Read whether we should export this object.
            ch = 'f';
            value = attrib.getValue();
            if (value != null)
                value = value.trim();
            if (value.length() > 0)
                ch = value.charAt(0);
            fsObj.setToExport(ch == 't' || ch == 'y' || ch == '1');
        }
        else
            fsObj.setToExport(false);
        // Read the export quality.
        attrib = fsElement.getAttributeNode(ATTR_EXPORT_QUALITY);
        if (attrib != null)
        {
            try
            {
                quality = Integer.parseInt(attrib.getValue());
                fsObj.setConversionQuality(quality);
            }
            catch (Exception ignored)
            {
            }
        }
        // Get the description strings.
        for (subNode = fsElement.getFirstChild(); subNode != null;
             subNode = subNode.getNextSibling())
        {
            if (!(subNode instanceof Element))
                continue;
            name = ((Element) subNode).getTagName();
            if (name.equals(ELEM_TITLE))
                fsObj.set(AbstractFSObject.TITLE, getTextContent(subNode));
            else if (name.equals(ELEM_SUBTITLE))
                fsObj.set(AbstractFSObject.SUBTITLE, getTextContent(subNode));
            else if (name.equals(ELEM_DESCRIPTION))
                fsObj.set(AbstractFSObject.DESCRIPTION, getTextContent(subNode));
            else if (name.equals(ELEM_LOCATION))
                fsObj.set(AbstractFSObject.LOCATION, getTextContent(subNode));
            else if (name.equals(ELEM_REMARK))
                fsObj.set(AbstractFSObject.REMARK, getTextContent(subNode));
            else if (name.equals(ELEM_PROPERTY))
            {
                name = ((Element) subNode).getAttributeNode(ATTR_NAME).
                    getValue();
                fsObj.set(name, getTextContent(subNode));
            }
        }
    } // readDescription

    /**
     * Private methode to get the text from a node (Node.getTextContent() is not
     * used here as it is only available in Java 1.5).
     * @param node - The node whose text is needed
     * @return The text
     */
    private static String getTextContent (Node node)
    {
        NodeList     children;
        Node         child;
        String       value;
        StringBuffer text;

        // We need to retrieve the text from elements, entity references, CDATA
        // sections, and text nodes; but not comments or processing instructions
        int type = node.getNodeType();
        if (type == Node.COMMENT_NODE ||
            type == Node.PROCESSING_INSTRUCTION_NODE)
        {
            return "";
        }
        text = new StringBuffer();
        value = node.getNodeValue();
        if (value != null)
            text.append(value);
        if (node.hasChildNodes())
        {
            children = node.getChildNodes();
            for (int i = 0; i < children.getLength(); i++)
            {
                child = children.item(i);
                text.append(getTextContent(child));
            }
        }
        return text.toString().trim();
    } // getTextContent

    /**
     * @see com.dgrossmann.photo.dir.persist.AbstractDirPersist#internalSave(com.dgrossmann.photo.dir.DirectoryObject, java.lang.String, boolean)
     */
    @Override
	protected void internalSave
        ( DirectoryObject seriesDirObj
        , String          metadataStoreName
        , boolean         bForWeb
        ) throws PersistException
    {
        File                   metaDataFile;
        PrintWriter            writer;
        DocumentBuilderFactory fac;
        DocumentBuilder        builder;
        Document               doc;
        Element                rootElem;
        Comment                comment;
        DateFormat             df;

        metaDataFile = new File(metadataStoreName);
        writer = null;
        try
        {
            fac = DocumentBuilderFactory.newInstance();
            fac.setValidating(false);
            builder = fac.newDocumentBuilder();
            doc = builder.newDocument();
            // Add the title comments.
            comment = doc.createComment(AppInfo.APP_NAME + " (Version "
                + AppInfo.getVersionString() + ")"
                + (bForWeb? " Web-exported" : "")
                + " photo series metadata file.");
            doc.appendChild(comment);
            df = DateFormat.getDateTimeInstance
                (DateFormat.LONG, DateFormat.MEDIUM);
            comment = doc.createComment("Created: " + df.format(new Date()));
            doc.appendChild(comment);
            if (bForWeb)
            {
                comment = doc.createComment("Metadata exported for the Web. "
                    + "Do not load as regular series metadata.");
                doc.appendChild(comment);
            }
            // Create the series element.
            rootElem = doc.createElementNS(DEFAULT_NAMESPACE, ELEM_SERIES);
            doc.appendChild(rootElem);
            if (bForWeb)
                rootElem.setAttribute(ATTR_WEB_EXPORTED, "yes");
            this.writeDescription(seriesDirObj, bForWeb, rootElem, doc);
            // Write the series contents into the XML document.
            this.writeGroupBody(seriesDirObj, bForWeb, rootElem, doc);
            // Save the completed XML document.
            TransformerFactory xformFactory = TransformerFactory.newInstance();
            Transformer xForm = xformFactory.newTransformer();
            xForm.setOutputProperty(OutputKeys.ENCODING, bForWeb ?
                m_exportXmlEncoding : m_saveXmlEncoding);
            xForm.setOutputProperty(OutputKeys.INDENT, "yes");
            Source input = new DOMSource(doc);
            Result output = new StreamResult(new PrintWriter
                (new FileOutputStream(metaDataFile)));
            xForm.transform(input, output);
        }
        catch (Exception exc)
        {
            PersistException pe = new PersistException(
                PersistException.E_SAVE_SERIES, seriesDirObj,
                metaDataFile.getAbsolutePath(), exc);
            pe.log();
            throw pe;
        }
        finally
        {
            if (writer != null)
                writer.close();
        }
    } // internalSave

    /**
     * Private helper for <tt>internalSave</tt> to write the group body
     * (without directory name, description and end marker).
     * @param dirObj - Directory object whose contained files and subdirectories
     * should be saved
     * @param bForWeb - <tt>True</tt> to write for web export
     * @param dirElem - XML element representing the directory
     * @param doc - The XML document
     * @throws Exception on error
     */
    private void writeGroupBody
        ( DirectoryObject dirObj
        , boolean         bForWeb
        , Element         dirElem
        , Document        doc
        ) throws Exception
    {
        Iterator<DirectoryObject> subDirIter;
        Iterator<FileObject>      fileIter;
        DirectoryObject           subDir;
        FileObject                fileObj;
        Element                   contentsElem, elem;

        contentsElem = doc.createElement(ELEM_CONTENTS);
        dirElem.appendChild(contentsElem);
        subDirIter = dirObj.getSubDirIterator();
        while (subDirIter.hasNext())
        {
            subDir = subDirIter.next();
            if (bForWeb && !subDir.isToExport())
                continue;
            elem = doc.createElement(ELEM_GROUP);
            contentsElem.appendChild(elem);
            this.writeDescription(subDir, bForWeb, elem, doc);
            this.writeGroupBody(subDir, bForWeb, elem, doc);
        }
        fileIter = dirObj.getFileIterator();
        while (fileIter.hasNext())
        {
            fileObj = fileIter.next();
            if (bForWeb && !fileObj.isToExport())
                continue;
            elem = doc.createElement(fileObj.isSeparator() ?
                ELEM_SEPARATOR : ELEM_FILE);
            contentsElem.appendChild(elem);
            this.writeDescription(fileObj, bForWeb, elem, doc);
        }
    } // writeGroupBody

    /**
     * Private helper for <tt>internalSave</tt> to write the properties of a
     * file system object.
     * @param fsObj - File system object
     * @param bForWeb - <tt>True</tt> to write for web export
     * @param fsObjElem XML element representing the file system object
     * @param doc - The XML document
     * @throws Exception on error
     */
    private void writeDescription
        ( AbstractFSObject fsObj
        , boolean          bForWeb
        , Element          fsObjElem
        , Document         doc
        ) throws Exception
    {
        String[] propNames;
        String   name, val;
        Element  elem;
        int      i;

        // Prepare and add the name attribute.
        if (fsObj instanceof DirectoryObject)
            name = fsObj.getFileName();
        else if (fsObj.isReference())
            name = fsObj.getFileNamePart();
        else if (bForWeb)
        {
            FileObject fileObj = (FileObject) fsObj;
            if (fileObj.getFileType() != FileObject.TYPE_IMAGE_PREVIEW)
                name = fileObj.getFileName();
            else
            {
                // Change for other JPEG image types.
                name = fileObj.getBaseFileName() + ".jpg";
            }
            name = name.toLowerCase();
        }
        else
            name = fsObj.getFileName();
        if (name.length() == 0)
            name = fsObj.getFileNamePart();
        if (bForWeb)
            name = name.toLowerCase();
        fsObjElem.setAttribute(ATTR_NAME, name);
        // Write the file system object properties.
        propNames = fsObj.getUsedPropertyNames();
        for (i = 0; i < propNames.length; i++)
        {
            val = fsObj.getTransformed(propNames[i], true);
            if (val == null || val.trim().length() == 0)
                continue;
            if (propNames[i].equals(AbstractFSObject.DATE_BEGIN))
                fsObjElem.setAttribute(ATTR_BEGIN_DATE, val);
            else if (propNames[i].equals(AbstractFSObject.DATE_END))
                fsObjElem.setAttribute(ATTR_END_DATE, val);
            else if (propNames[i].equals(AbstractFSObject.HREF))
                fsObjElem.setAttribute(ATTR_HREF, val);
            else if (propNames[i].equals(AbstractFSObject.TITLE))
            {
                elem = doc.createElement(ELEM_TITLE);
                elem.appendChild(doc.createCDATASection(val));
                fsObjElem.appendChild(elem);
            }
            else if (propNames[i].equals(AbstractFSObject.SUBTITLE))
            {
                elem = doc.createElement(ELEM_SUBTITLE);
                elem.appendChild(doc.createCDATASection(val));
                fsObjElem.appendChild(elem);
            }
            else if (propNames[i].equals(AbstractFSObject.DESCRIPTION))
            {
                elem = doc.createElement(ELEM_DESCRIPTION);
                elem.appendChild(doc.createCDATASection(val));
                fsObjElem.appendChild(elem);
            }
            else if (propNames[i].equals(AbstractFSObject.LOCATION))
            {
                elem = doc.createElement(ELEM_LOCATION);
                elem.appendChild(doc.createCDATASection(val));
                fsObjElem.appendChild(elem);
            }
            else if (propNames[i].equals(AbstractFSObject.REMARK))
            {
                elem = doc.createElement(ELEM_REMARK);
                elem.appendChild(doc.createCDATASection(val));
                fsObjElem.appendChild(elem);
            }
            else
            {
                elem = doc.createElement(ELEM_PROPERTY);
                elem.setAttribute(ATTR_NAME, propNames[i]);
                elem.appendChild(doc.createCDATASection(val));
                fsObjElem.appendChild(elem);
            }
        }
        // Write the fixed properties.
        if (fsObj.isReference())
            fsObjElem.setAttribute(ATTR_REFERENCE, "yes");
        if (bForWeb)
        {
            // Write the guessed title if empty.
            val = fsObj.getTitle(false);
            val = AbstractFSObject.transformAccents(val, true);
            if (val.length() == 0)
            {
                elem = doc.createElement(ELEM_TITLE);
                val = fsObj.getTitle(true);
                val = AbstractFSObject.transformAccents(val, true);
                elem.appendChild(doc.createCDATASection(val));
                fsObjElem.appendChild(elem);
            }
        }
        else
        {
            // Write the export settings only when not exporting.
            if (fsObj.isToExport())
            {
                fsObjElem.setAttribute(ATTR_EXPORT, "yes");
                if (fsObj.getConversionQuality() > 0)
                {
                    fsObjElem.setAttribute(ATTR_EXPORT_QUALITY,
                        Integer.toString(fsObj.getConversionQuality()));
                }
            }
            else
                fsObjElem.setAttribute(ATTR_EXPORT, "no");
        }
    } // writeDescription
} // DirPersistXmlFile
