/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
  Interface: PersistException
    Created: 2 January, 2003
        $Id: PersistException.java 42 2006-01-21 17:47:48Z Dirk $
  $Revision: 42 $
      $Date: 2006-01-21 18:47:48 +0100 (Sa, 21 Jan 2006) $
    $Author: Dirk $
===============================================================================
*/

package com.dgrossmann.photo.dir.persist;

import com.dgrossmann.photo.dir.DirectoryObject;

/**
 * Represents a series or settings load/save exception.
 */
public class PersistException extends Exception
{
    public static final int E_UNKNOWN        = 0;
    public static final int E_LOAD_SERIES    = 1;
    public static final int E_LOAD_SETTINGS  = 2;
    public static final int E_SAVE_SERIES    = 3;
    public static final int E_SAVE_SETTINGS  = 4;
    public static final int E_ADD_CONTAINER  = 5;
    
    private int       m_errorID;
    private Object    m_object;
    private String    m_metaDataFileName;
    private String    m_msg;
    private Exception m_e;
    
    /**
     * Creates a new instance of <tt>PersistException</tt>
     * @param errorID - Error identifier
     * @param object - Object causing the exception
     * @param metaDataFileName - Name of the metadata file
     * @param msg - String message
     */
    public PersistException
        ( int       errorID
        , Object    object
        , String    metaDataFileName
        , String    msg
        )
    {
        m_errorID = errorID;
        m_object = object;
        m_metaDataFileName = metaDataFileName;
        m_msg = msg;
        m_e = null;
        if (m_object instanceof DirectoryObject)
            ((DirectoryObject) m_object).setError();
    } // PersistException
    
    /**
     * Creates a new instance of <tt>PersistException</tt>
     * @param errorID - Error identifier
     * @param object - Object causing the exception
     * @param metaDataFileName - Name of the metadata file
     * @param e - Exception to be wrapped
     */
    public PersistException
        ( int       errorID
        , Object    object
        , String    metaDataFileName
        , Exception e
        )
    {
        m_errorID = errorID;
        m_object = object;
        m_metaDataFileName = metaDataFileName;
        m_msg = null;
        m_e = e;
        if (m_object instanceof DirectoryObject)
            ((DirectoryObject) m_object).setError();
    } // PersistException
    
    /**
     * Gets the exception message.
     * @return Exception message string
     */
    public String getMessage ()
    {
        String msg;
        
        if (m_errorID == E_ADD_CONTAINER)
        {
            msg = "Cannot add series container for \"" + m_object.toString()
                + "\"";
        }
        else if (m_object instanceof DirectoryObject)
        {
            msg = "Cannot " + (m_errorID == E_LOAD_SERIES ? "load" : "save")
                + " series \"" + m_object + "\"";
            if (m_metaDataFileName != null && m_metaDataFileName.length() > 0)
                msg += " using metadata file \"" + m_metaDataFileName + "\"";
        }
        else
        {
            msg = "Cannot "
                + (m_errorID == E_LOAD_SERIES ||
                   m_errorID == E_LOAD_SETTINGS ? "load" : "save")
                + " object \"" + m_object.toString();
            if (m_metaDataFileName != null && m_metaDataFileName.length() > 0)
                msg += " using file \"" + m_metaDataFileName + "\"";
        }
        if (m_msg != null && m_msg.length() > 0)
            msg += '\n' + msg;
        else if (m_e != null)
            msg += '\n' + m_e.toString();
        return msg;
    } // getMessage
    
    /** Logs the exception on standard out. */
    public void log ()
    {
        System.err.println("E: " + this.getMessage());
    } // log
} // PersistException
