/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: CDateChooser
    Created: 5 January, 2003
        $Id: DateChooserDialog.java 41 2006-01-14 15:49:02Z Dirk $
  $Revision: 41 $
      $Date: 2006-01-14 16:49:02 +0100 (Sa, 14 Jan 2006) $
    $Author: Dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui.dialog;

import java.awt.Dialog;
import java.awt.Frame;
import java.awt.Rectangle;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.Calendar;
import java.util.Date;
import java.util.StringTokenizer;

import javax.swing.JDialog;
import javax.swing.JOptionPane;

import com.dgrossmann.photo.AppInfo;

/**
 * Dialog class that allows the user to choose a date.
 */
public class DateChooserDialog extends JDialog implements ItemListener
{
    private static final String	SEL_SPACER	= "----------";

	private Rectangle m_parentRect;
    private String    m_dlgTitle;
    private boolean   m_bIsChanged;
    
    private int       m_day, m_month, m_year;
    
    private static final String[] monthNames =
    {
        "January", "February", "March", "April", "May", "June", "July",
        "August", "September", "October", "November", "December"
    };
    
    private static final int[] daysPerMonth =
    {
        31, 29, 31, 30, 31, 30, 31,
        31, 30, 31, 30, 31
    };
    
    /**
     * Creates a new <tt>CDateChooser</tt> form instance.
     * @param parent - Parent frame in the user interface
     * @param dateValue - Date in the form "[YYYY[-MM[-DD]]]"
     * @param dlgTitle - Title of the dialog window
     * @param modal - <tt>True</tt> for a modal dialog
     */
    public DateChooserDialog(Frame parent, String dateValue, String dlgTitle, boolean modal)
    {
        super(parent, modal);
        m_parentRect = parent.getBounds();
        m_dlgTitle = dlgTitle;
        this.initialize(dateValue);
    } // CDateChooser
    
    /**
     * Creates a new <tt>CDateChooser</tt> form instance.
     * @param parent - Parent dialog in the user interface
     * @param dateValue - Date in the form "[YYYY[-MM[-DD]]]"
     * @param dlgTitle - Title of the dialog window
     * @param modal - <tt>True</tt> for a modal dialog
     */
    public DateChooserDialog(Dialog parent, String dateValue, String dlgTitle, boolean modal)
    {
        super(parent, modal);
        m_parentRect = parent.getBounds();
        m_dlgTitle = dlgTitle;
        this.initialize(dateValue);
    } // CDateChooser
    
    /* Initializes the dialog instance */
    private void initialize(String dateValue)
    {
        initComponents();
        // Set the instance data.
        m_bIsChanged = false;
        m_day = m_month = m_year = 0;
        if (dateValue != null && dateValue.length() > 0)
        {
            StringTokenizer tokens = new StringTokenizer(dateValue, "-");
            try
            {
                if (tokens.hasMoreTokens())
                    m_year = Integer.parseInt(tokens.nextToken());
                if (tokens.hasMoreTokens())
                    m_month = Integer.parseInt(tokens.nextToken());
                if (tokens.hasMoreTokens())
                    m_day = Integer.parseInt(tokens.nextToken());
            }
            catch (Exception ex)
            {
            }
        }
        // Set the title.
        if (m_dlgTitle.equalsIgnoreCase("begin"))
            this.setTitle("Select the Begin Date");
        else if (m_dlgTitle.equalsIgnoreCase("end"))
            this.setTitle("Select the End Date");
        else if (m_dlgTitle.length() == 0)
            this.setTitle("Select Date");
        else
            this.setTitle(m_dlgTitle);
        // Set the day and month values.
        int i;
        dayComboBox.addItem(SEL_SPACER);
        for (i = 1; i <= 31; i++)
            dayComboBox.addItem(new Integer(i));
        if (m_day > 0)
            dayComboBox.setSelectedIndex(m_day);
        monthComboBox.addItem(SEL_SPACER);
        for (i = 0; i < monthNames.length; i++)
            monthComboBox.addItem(monthNames[i]);
        if (m_month > 0)
            monthComboBox.setSelectedIndex(m_month);
        monthComboBox.addItemListener(this);
        // Set the year value.
        Integer yearInt;
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        int firstYear = cal.get(Calendar.YEAR) - 20;
        if (m_year > 0 && (m_year < firstYear ||
            m_year > cal.get(Calendar.YEAR)))
        {
            yearComboBox.addItem(new Integer(m_year));
        }
        yearComboBox.addItem(SEL_SPACER);
        for (i = cal.get(Calendar.YEAR); i >= firstYear; i--)
        {
            yearInt = new Integer(i);
            yearComboBox.addItem(yearInt);
            if (m_year == i)
                yearComboBox.setSelectedItem(yearInt);
        }
        // Position this dialog centered in the parent dialog.
        Rectangle thisRect = this.getBounds();
        thisRect.x = m_parentRect.x +
            (m_parentRect.width - thisRect.width) / 2;
        thisRect.y = m_parentRect.y +
            (m_parentRect.height - thisRect.height) / 2;
        if (thisRect.x < 0)
            thisRect.x = 0;
        if (thisRect.y < 0)
            thisRect.y = 0;
        this.setBounds(thisRect);
    } // CDateChooser
    
    /**
     * Shows the dialog window.
     * @return <tt>True</tt> if the user made changes, <tt>false</tt> if she
     * canceled the dialog
     */
    public boolean showDialog ()
    {
        this.setVisible(true);
        return m_bIsChanged;
    } // showDialog;
    
    /** Gets the date string from the values selected by the user. */
    public String getDateString ()
    {
        String str = "";

        if (m_year > 0)
        {
            str = Integer.toString(m_year);
            if (m_month >= 1 && m_month <= 12)
            {
                str += "-" + m_month;
                if (m_day > 0 && m_day <= daysPerMonth[m_month-1])
                    str += "-" + m_day;
            }
        }
        return str;
    } // getDateString
    
    /* Listener for the combo box selection change events. */
    public void itemStateChanged (ItemEvent e)
    {
        if (e.getSource() == monthComboBox)
        {
            //Object val = monthComboBox.getSelectedItem();
        }
    } // itemStateChanged
    
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    private void initComponents()//GEN-BEGIN:initComponents
    {
        java.awt.GridBagConstraints gridBagConstraints;

        contentPanel = new javax.swing.JPanel();
        titleLabel = new javax.swing.JLabel();
        titleLabel2 = new javax.swing.JLabel();
        dayLabel = new javax.swing.JLabel();
        monthLabel = new javax.swing.JLabel();
        yearLabel = new javax.swing.JLabel();
        dayComboBox = new javax.swing.JComboBox();
        slashLabel = new javax.swing.JLabel();
        monthComboBox = new javax.swing.JComboBox();
        slashLabel2 = new javax.swing.JLabel();
        yearComboBox = new javax.swing.JComboBox();
        clearDateButton = new javax.swing.JButton();
        spacerLabel = new javax.swing.JLabel();
        buttonPanel = new javax.swing.JPanel();
        okButton = new javax.swing.JButton();
        cancelButton = new javax.swing.JButton();

        getContentPane().setLayout(new java.awt.GridBagLayout());

        setTitle("Select Date");
        setResizable(false);
        addWindowListener(new java.awt.event.WindowAdapter()
        {
            public void windowClosing(java.awt.event.WindowEvent evt)
            {
                closeDialog(evt);
            }
        });

        contentPanel.setLayout(new java.awt.GridBagLayout());

        contentPanel.setMinimumSize(new java.awt.Dimension(360, 118));
        contentPanel.setPreferredSize(new java.awt.Dimension(360, 118));
        titleLabel.setText("Please enter the date - you may leave the day and month");
        titleLabel.setMaximumSize(new java.awt.Dimension(3660, 17));
        titleLabel.setMinimumSize(new java.awt.Dimension(500, 17));
        titleLabel.setPreferredSize(new java.awt.Dimension(500, 17));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 7;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        contentPanel.add(titleLabel, gridBagConstraints);

        titleLabel2.setText("empty (fill from right to left) or clear the date altogether.");
        titleLabel2.setVerticalAlignment(javax.swing.SwingConstants.TOP);
        titleLabel2.setMaximumSize(new java.awt.Dimension(2810, 17));
        titleLabel2.setMinimumSize(new java.awt.Dimension(280, 17));
        titleLabel2.setPreferredSize(new java.awt.Dimension(280, 17));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 7;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 12, 6);
        contentPanel.add(titleLabel2, gridBagConstraints);

        dayLabel.setText("Day:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        contentPanel.add(dayLabel, gridBagConstraints);

        monthLabel.setText("Month:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 0);
        contentPanel.add(monthLabel, gridBagConstraints);

        yearLabel.setText("Year:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 0);
        contentPanel.add(yearLabel, gridBagConstraints);

        dayComboBox.setEditable(true);
        dayComboBox.setToolTipText("Day");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(3, 0, 0, 0);
        contentPanel.add(dayComboBox, gridBagConstraints);

        slashLabel.setText("/");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.insets = new java.awt.Insets(3, 2, 0, 0);
        contentPanel.add(slashLabel, gridBagConstraints);

        monthComboBox.setEditable(true);
        monthComboBox.setToolTipText("Month");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(3, 6, 0, 0);
        contentPanel.add(monthComboBox, gridBagConstraints);

        slashLabel2.setText("/");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.insets = new java.awt.Insets(3, 2, 0, 0);
        contentPanel.add(slashLabel2, gridBagConstraints);

        yearComboBox.setEditable(true);
        yearComboBox.setToolTipText("Year");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(3, 6, 0, 0);
        contentPanel.add(yearComboBox, gridBagConstraints);

        clearDateButton.setText("Clear");
        clearDateButton.setToolTipText("Clear the date setting");
        clearDateButton.setMargin(new java.awt.Insets(2, 2, 2, 2));
        clearDateButton.setMaximumSize(new java.awt.Dimension(64, 25));
        clearDateButton.setMinimumSize(new java.awt.Dimension(64, 25));
        clearDateButton.setPreferredSize(new java.awt.Dimension(64, 25));
        clearDateButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                clearDateButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(3, 6, 0, 0);
        contentPanel.add(clearDateButton, gridBagConstraints);

        spacerLabel.setText("  ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 6;
        gridBagConstraints.gridy = 3;
        contentPanel.add(spacerLabel, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.weighty = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 0, 6);
        getContentPane().add(contentPanel, gridBagConstraints);

        buttonPanel.setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.RIGHT));

        okButton.setText("OK");
        okButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                okButtonActionPerformed(evt);
            }
        });

        buttonPanel.add(okButton);

        cancelButton.setText("Cancel");
        cancelButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                cancelButtonActionPerformed(evt);
            }
        });

        buttonPanel.add(cancelButton);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 3, 0);
        getContentPane().add(buttonPanel, gridBagConstraints);

        pack();
    }//GEN-END:initComponents

    private void clearDateButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_clearDateButtonActionPerformed
    {//GEN-HEADEREND:event_clearDateButtonActionPerformed
        m_day = m_month = m_year = 0;
        dayComboBox.setSelectedItem(SEL_SPACER);
        monthComboBox.setSelectedItem(SEL_SPACER);
        yearComboBox.setSelectedItem(SEL_SPACER);
    }//GEN-LAST:event_clearDateButtonActionPerformed

    private void cancelButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cancelButtonActionPerformed
        this.setVisible(false);
        this.dispose();
    }//GEN-LAST:event_cancelButtonActionPerformed

    private void okButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_okButtonActionPerformed
        Object  val;
        int     i;
        boolean bFound;
        
        // Verify the day of month.
        val = dayComboBox.getSelectedItem();
        if (val != null && val.toString().startsWith("-"))
            m_day = 0;
        else
        {
            try
            {
                m_day = Integer.parseInt(val.toString());
            }
            catch (Exception dayEx)
            {
                JOptionPane.showMessageDialog(this,
                    "Please enter a valid day of month number",
                    AppInfo.APP_NAME, JOptionPane.WARNING_MESSAGE);
                return;
            }
        }
        // Verify the month.
        val = monthComboBox.getSelectedItem();
        if (val != null && val.toString().startsWith("-"))
            m_month = m_day = 0;
        else
        {
            // Is it a month name ?-
            bFound = false;
            for (i = 0; i < monthNames.length; i++)
            {
                if (val.toString().equalsIgnoreCase(monthNames[i]))
                {
                    m_month = i + 1;
                    bFound = true;
                }
            }
            if (!bFound)
            {
                try
                {
                    m_month = Integer.parseInt(val.toString());
                    if (m_month < 1 || m_month > 12)
                        throw new Exception("");
                }
                catch (Exception dayEx)
                {
                    JOptionPane.showMessageDialog(this,
                        "Please enter a valid month name or number",
                        AppInfo.APP_NAME, JOptionPane.WARNING_MESSAGE);
                    return;
                }
            }
        }
        // Verify the day of month.
        if (m_month > 0 && m_day > daysPerMonth[m_month-1])
        {
            JOptionPane.showMessageDialog(this, "Month "
                + monthNames[m_month-1] + " does not have " + m_day + " days",
                AppInfo.APP_NAME, JOptionPane.WARNING_MESSAGE);
            return; 
        }
        // Verify the year.
        val = yearComboBox.getSelectedItem();
        if (val != null && val.toString().startsWith("-"))
            m_year = m_month = m_day = 0;
        else
        {
            try
            {
                m_year = Integer.parseInt(val.toString());
            }
            catch (Exception dayEx)
            {
                JOptionPane.showMessageDialog(this,
                    "Please enter a valid year",
                    AppInfo.APP_NAME, JOptionPane.WARNING_MESSAGE);
                return;
            }
        }
        // Close the dialog.
        m_bIsChanged = true;
        this.cancelButtonActionPerformed(evt);
    }//GEN-LAST:event_okButtonActionPerformed
    
    /** Closes the dialog */
    private void closeDialog(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_closeDialog
        this.setVisible(false);
        this.dispose();
    }//GEN-LAST:event_closeDialog
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JPanel buttonPanel;
    private javax.swing.JButton cancelButton;
    private javax.swing.JButton clearDateButton;
    private javax.swing.JPanel contentPanel;
    private javax.swing.JComboBox dayComboBox;
    private javax.swing.JLabel dayLabel;
    private javax.swing.JComboBox monthComboBox;
    private javax.swing.JLabel monthLabel;
    private javax.swing.JButton okButton;
    private javax.swing.JLabel slashLabel;
    private javax.swing.JLabel slashLabel2;
    private javax.swing.JLabel spacerLabel;
    private javax.swing.JLabel titleLabel;
    private javax.swing.JLabel titleLabel2;
    private javax.swing.JComboBox yearComboBox;
    private javax.swing.JLabel yearLabel;
    // End of variables declaration//GEN-END:variables
} // CDateChooser
