/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: SettingsDialog
    Created: 2 January, 2003
        $Id: SettingsDialog.java 160 2009-05-31 07:57:29Z dirk $
  $Revision: 160 $
      $Date: 2009-05-31 09:57:29 +0200 (So, 31 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui.dialog;

import java.awt.Color;
import java.awt.Font;
import java.awt.Rectangle;
import java.io.File;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import com.dgrossmann.photo.AppInfo;
import com.dgrossmann.photo.dir.SeriesContainer;
import com.dgrossmann.photo.settings.Settings;
import com.dgrossmann.photo.ui.ExplorerMainFrame;
import com.dgrossmann.photo.ui.panel.DirTreePanel;
import com.dgrossmann.photo.ui.panel.contents.ContentsPanel;
import com.dgrossmann.photo.webexport.IWebExport;
import com.dgrossmann.photo.webexport.ImageMagickExporter;

/**
 * The <i>Series and Web export Settings</i> dialog class.
 */
public class SettingsDialog extends JDialog
    implements ListSelectionListener, ChangeListener
{
    private ExplorerMainFrame m_frame;
    private Settings          m_settings;
    private SeriesContainer   m_seriesContainer;
    private Vector<String>    m_seriesContainerNames;
    private String            m_currentSContainerDir;
    private boolean           m_isChanged, m_bHaveImageMagick;

    /**
     * Creates a new <tt>SettingsDialog</tt> form instance.
     * @param frame - Parent frame in the user interface
     * @param settings - Settings object used to load and store the settings.
     * There is exactly one instance per application.
     * @param seriesContainer - Series container object that manages series
     * directories. There is exactly one instance per application.
     * @param modal - <tt>True</tt> for a modal dialog
     */
    public SettingsDialog
        ( ExplorerMainFrame frame
        , SeriesContainer   seriesContainer
        , Settings          settings
        , boolean           modal
        )
    {
        super(frame, modal);
        this.initComponents();
        // Fill the members.
        m_frame = frame;
        m_settings = settings;
        m_seriesContainer = seriesContainer;
        m_isChanged = false;
        m_currentSContainerDir = null;
        // Override font properties
        restartLabel.setFont(restartLabel.getFont().deriveFont(Font.BOLD));
        // Set the series container directories.
        m_seriesContainerNames = new Vector<String>(20);
        Iterator<String> sIter = m_settings.getSeriesContainerNameIterator();
        while (sIter.hasNext())
            m_seriesContainerNames.add(sIter.next());
        sContainerDirList.setListData(m_seriesContainerNames);
        sContainerDirList.setSelectionMode
            (ListSelectionModel.SINGLE_SELECTION);
        sContainerDirList.addListSelectionListener(this);
        // Set the structure properties.
        sTitleText.setText(m_settings.get(Settings.GALLERY_TITLE));
        sDescTextArea.setText(m_settings.get(Settings.GALLERY_DESCRIPTION));
        sDescTextArea.select(0, 0);
        String text = m_settings.get(Settings.GALLERY_COPYRIGHT);
        if (text == null || text.length() == 0)
        	text = Settings.DEFAULT_COPYRIGHT_STRING;
        sCopyrightText.setText(text);
        // Set the Web export directory.
        exportDirText.setText(m_settings.get(Settings.EXPORT_DIRECTORY));
        // Set the main image size properties.
        mainLengthText.setText
            (m_settings.get(Settings.EXPORT_MAINIMG_LENGTH,
            Integer.toString(Settings.EXPDEFAULT_MAINIMG_LENGTH)));
        mainMaxWidthText.setText
            (m_settings.get(Settings.EXPORT_MAINIMG_MAX_WIDTH,
            Integer.toString(Settings.EXPDEFAULT_MAINIMG_MAX_WIDTH)));
        mainMaxHeightText.setText
            (m_settings.get(Settings.EXPORT_MAINIMG_MAX_HEIGHT,
            Integer.toString(Settings.EXPDEFAULT_MAINIMG_MAX_HEIGHT)));
        mainQualityText.setText
            (m_settings.get(Settings.EXPORT_MAINIMG_QUALITY,
            Integer.toString(Settings.EXPDEFAULT_MAINIMG_QUALITY)));
        // Set the index image size properties.
        indexLengthText.setText
            (m_settings.get(Settings.EXPORT_INDEXIMG_LENGTH,
            Integer.toString(Settings.EXPDEFAULT_INDEXIMG_LENGTH)));
        indexQualityText.setText
            (m_settings.get(Settings.EXPORT_INDEXIMG_QUALITY,
            Integer.toString(Settings.EXPDEFAULT_INDEXIMG_QUALITY)));
        // Set whether we use ImageMagick.
        IWebExport magickExporter = new ImageMagickExporter();
        m_bHaveImageMagick = false;
        if (magickExporter.isAvailable())
        {
            m_bHaveImageMagick = true;
            useMagickCheckBox.setSelected
                (m_settings.getInt(Settings.EXPORT_USE_IMAGE_MAGICK, 0) > 0);
        }
        else
        {
            useMagickCheckBox.setSelected(false);
            useMagickCheckBox.setEnabled(false);
            useMagickDescLabel.setText("(ImageMagick is not installed)");
            useMagickDescLabel.setForeground(new Color(0, 70, 213));
        }
        // Set the save options.
        saveAsXMLCheckBox.setSelected(m_settings.getBoolean
            (Settings.METADATA_SAVE_XML, true));
        saveAsXMLCheckBox.addChangeListener(this);
        if (saveAsXMLCheckBox.isSelected())
        {
            String enc = m_settings.get(Settings.METADATA_SAVE_ENCODING);
            saveEncodingTextField.setText((enc.length() > 0) ? enc :
                Settings.METADEFAULT_XML_ENCODING);
        }
        else
            saveEncodingTextField.setEnabled(false);
        exportAsXmlCheckBox.setSelected(m_settings.getBoolean
            (Settings.METADATA_EXPORT_XML, true));
        exportAsXmlCheckBox.addChangeListener(this);
        if (saveAsXMLCheckBox.isSelected())
        {
            String enc = m_settings.get(Settings.METADATA_EXPORT_ENCODING);
            exportEncodingTextField.setText((enc.length() > 0) ? enc :
                Settings.METADEFAULT_XML_ENCODING);
        }
        else
            exportEncodingTextField.setEnabled(false);
        // Set the panel options.
        // For the directory tree panel.
        saveCurrentDirCheckBox.setSelected(m_settings.getBoolean
            (DirTreePanel.SAVE_CURRENT_DIR_ON_EXIT, false));
        // For the directory contents panel.
        doubleClickOpensDirCheckBox.setSelected(m_settings.getBoolean
            (ContentsPanel.DBL_CLICK_DIR_OPENS, false));
        doubleClickOpensFileCheckBox.setSelected(m_settings.getBoolean
            (ContentsPanel.DBL_CLICK_FILE_OPENS, false));
        // Disable the buttons.
        delButton.setEnabled(false);
        moveUpButton.setEnabled(false);
        moveDownButton.setEnabled(false);
        // Position centered in the parent dialog.
        Rectangle parentRect, thisRect;
        parentRect = frame.getBounds();
        thisRect = this.getBounds();
        thisRect.x = parentRect.x + (parentRect.width - thisRect.width) / 2;
        thisRect.y = parentRect.y + (parentRect.height - thisRect.height) / 2;
        if (thisRect.x < 0)
            thisRect.x = 0;
        if (thisRect.y < 0)
            thisRect.y = 0;
        this.setBounds(thisRect);
    } // SettingsDialog

    /**
     * Shows the dialog window.
     * @return <tt>True</tt> if the user made changes, <tt>false</tt> if she
     * canceled the dialog
     */
    public boolean showDialog ()
    {
        this.setVisible(true);
        return m_isChanged;
    } // showDialog

    /**
     * List selection listener event handler.
     * @see javax.swing.event.ListSelectionListener#valueChanged(javax.swing.event.ListSelectionEvent)
     */
    public void valueChanged (ListSelectionEvent listSelectionEvent)
    {
        boolean bHasNewSelection = false;

        if (listSelectionEvent.getValueIsAdjusting())
            return;
        m_currentSContainerDir = null;
        Object val = sContainerDirList.getSelectedValue();
        if (val != null)
        {
            m_currentSContainerDir = (String) val;
            bHasNewSelection = true;
        }
        int index = sContainerDirList.getSelectedIndex();
        moveUpButton.setEnabled
            (index > 0 && index < m_seriesContainerNames.size());
        moveDownButton.setEnabled
            (index >= 0 && index < m_seriesContainerNames.size() - 1);
        delButton.setEnabled(bHasNewSelection);
    } // valueChanged

    /**
     * Check box change event handler for the metadata save boxes.
     * @see javax.swing.event.ChangeListener#stateChanged(javax.swing.event.ChangeEvent)
     */
    public void stateChanged (ChangeEvent e)
    {
        String text;

        if (e.getSource() == saveAsXMLCheckBox)
        {
            if (saveAsXMLCheckBox.isSelected())
            {
                // Activate and populate the edit control.
                saveEncodingTextField.setEnabled(true);
                text = saveEncodingTextField.getText();
                if (text == null || text.length() == 0)
                {
                    saveEncodingTextField.setText
                        (Settings.METADEFAULT_XML_ENCODING);
                }
            }
            else
            {
                saveEncodingTextField.setEnabled(false);
                text = saveEncodingTextField.getText();
                if (text != null &&
                    text.equalsIgnoreCase(Settings.METADEFAULT_XML_ENCODING))
                {
                    saveEncodingTextField.setText("");
                }
            }
        }
        else if (e.getSource() == exportAsXmlCheckBox)
        {
            if (exportAsXmlCheckBox.isSelected())
            {
                // Activate and populate the edit control.
                exportEncodingTextField.setEnabled(true);
                text = exportEncodingTextField.getText();
                if (text == null || text.length() == 0)
                {
                    exportEncodingTextField.setText
                        (Settings.METADEFAULT_XML_ENCODING);
                }
            }
            else
            {
                exportEncodingTextField.setEnabled(false);
                text = exportEncodingTextField.getText();
                if (text != null &&
                    text.equalsIgnoreCase(Settings.METADEFAULT_XML_ENCODING))
                {
                    exportEncodingTextField.setText("");
                }
            }
        }

    } // stateChanged

    /**
     * Private method to check an integer value whether it lies in a range.
     * @param textField - The text field containing the value
     * @param fieldDesc - Description string of the text field
     * @param tabIndex - Index of the tab containing the field
     * @param minValue - Minimum value
     * @param maxValue - Maximum value
     * @return The integer value on success. On error, <tt>null</tt> is returned
     * and the text field selected
     */
    private Integer validateInt
        ( JTextField textField
        , String     fieldDesc
        , int        tabIndex
        , int        minValue
        , int        maxValue
        )
    {
        String  s, msg;
        int     val;
        boolean bOK;

        bOK = true;
        val = -1;
        msg = "";
        s = textField.getText().trim();
        try
        {
            val = Integer.parseInt(s);
        }
        catch (Exception e)
        {
            bOK = false;
            msg = "Please enter a number in the field \"" + fieldDesc + "\"";
        }
        if (bOK && minValue <= maxValue && (val < minValue || val > maxValue))
        {
            bOK = false;
            msg = "The value of \"" + fieldDesc + "\" (" + val
                + ") has to be between " + minValue + " and " + maxValue;
        }
        if (bOK)
        	    return new Integer(val);
        JOptionPane.showMessageDialog(this, msg, AppInfo.APP_NAME,
            JOptionPane.WARNING_MESSAGE);
        jTabbedPane.setSelectedIndex(tabIndex);
        textField.requestFocus();
        return null;
    } // validateInt

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents()
    {
        java.awt.GridBagConstraints gridBagConstraints;

        jTabbedPane = new javax.swing.JTabbedPane();
        seriesPanel = new javax.swing.JPanel();
        listPanel = new javax.swing.JPanel();
        titleLabel = new javax.swing.JLabel();
        titleLabel1 = new javax.swing.JLabel();
        titleLabel2 = new javax.swing.JLabel();
        jScrollPane1 = new javax.swing.JScrollPane();
        sContainerDirList = new javax.swing.JList();
        addButton = new javax.swing.JButton();
        delButton = new javax.swing.JButton();
        moveUpButton = new javax.swing.JButton();
        moveDownButton = new javax.swing.JButton();
        sTitlePanel = new javax.swing.JPanel();
        jLabel4 = new javax.swing.JLabel();
        sTitleText = new javax.swing.JTextField();
        jLabel5 = new javax.swing.JLabel();
        jScrollPane2 = new javax.swing.JScrollPane();
        sDescTextArea = new javax.swing.JTextArea();
        jLabel1 = new javax.swing.JLabel();
        sCopyrightText = new javax.swing.JTextField();
        exportPanel = new javax.swing.JPanel();
        exportDirPanel = new javax.swing.JPanel();
        exportDirLabel = new javax.swing.JLabel();
        exportDirLabel1 = new javax.swing.JLabel();
        exportDirLabel2 = new javax.swing.JLabel();
        exportDirLabel3 = new javax.swing.JLabel();
        exportDirLabel4 = new javax.swing.JLabel();
        exportDirLabel5 = new javax.swing.JLabel();
        exportDirText = new javax.swing.JTextField();
        exportDirSelButton = new javax.swing.JButton();
        convSizePanel = new javax.swing.JPanel();
        mainImageLabel = new javax.swing.JLabel();
        mainLengthLabel = new javax.swing.JLabel();
        mainLengthText = new javax.swing.JTextField();
        mainMaxWidthLabel = new javax.swing.JLabel();
        mainMaxWidthText = new javax.swing.JTextField();
        mainMaxHeightLabel = new javax.swing.JLabel();
        mainMaxHeightText = new javax.swing.JTextField();
        mainQualityLabel = new javax.swing.JLabel();
        mainQualityText = new javax.swing.JTextField();
        percentLabel = new javax.swing.JLabel();
        jSeparator2 = new javax.swing.JSeparator();
        indexImageLabel = new javax.swing.JLabel();
        indexLengthLabel = new javax.swing.JLabel();
        indexLengthText = new javax.swing.JTextField();
        indexQualityLabel = new javax.swing.JLabel();
        indexQualityText = new javax.swing.JTextField();
        percentLabel1 = new javax.swing.JLabel();
        convToolPanel1 = new javax.swing.JPanel();
        useMagickCheckBox = new javax.swing.JCheckBox();
        useMagickDescLabel = new javax.swing.JLabel();
        savePanel = new javax.swing.JPanel();
        metadataFormatPanel = new javax.swing.JPanel();
        saveLabel = new javax.swing.JLabel();
        saveTitleLabel = new javax.swing.JLabel();
        saveAsXMLCheckBox = new javax.swing.JCheckBox();
        saveEncodingLabel = new javax.swing.JLabel();
        saveEncodingTextField = new javax.swing.JTextField();
        jSeparator3 = new javax.swing.JSeparator();
        exportLabel = new javax.swing.JLabel();
        exportTitleLabel = new javax.swing.JLabel();
        exportAsXmlCheckBox = new javax.swing.JCheckBox();
        exportEncodingLabel = new javax.swing.JLabel();
        exportEncodingTextField = new javax.swing.JTextField();
        jSeparator4 = new javax.swing.JSeparator();
        restartLabel = new javax.swing.JLabel();
        restartLabel2 = new javax.swing.JLabel();
        panelOptionsPanel = new javax.swing.JPanel();
        dirTreePanelOptionsPanel = new javax.swing.JPanel();
        saveCurrentDirCheckBox = new javax.swing.JCheckBox();
        contentsPanelOptionsPanel = new javax.swing.JPanel();
        doubleClickOpensDirCheckBox = new javax.swing.JCheckBox();
        doubleClickOpensFileCheckBox = new javax.swing.JCheckBox();
        buttonPanel = new javax.swing.JPanel();
        okButton = new javax.swing.JButton();
        cancelButton = new javax.swing.JButton();

        this.setTitle("Preferences");
        this.addWindowListener(new java.awt.event.WindowAdapter()
        {
            @Override
			public void windowClosing(java.awt.event.WindowEvent evt)
            {
                SettingsDialog.this.closeDialog(evt);
            }
        });
        this.getContentPane().setLayout(new java.awt.GridBagLayout());

        seriesPanel.setOpaque(false);
        seriesPanel.setLayout(new java.awt.GridBagLayout());

        listPanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Series Container Directories"));
        listPanel.setOpaque(false);
        listPanel.setLayout(new java.awt.GridBagLayout());

        titleLabel.setText("A \"series container\" is a directory that contains photo series. A photo series is a directory that");
        titleLabel.setMaximumSize(new java.awt.Dimension(534, 17));
        titleLabel.setMinimumSize(new java.awt.Dimension(534, 17));
        titleLabel.setPreferredSize(new java.awt.Dimension(534, 17));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 6);
        listPanel.add(titleLabel, gridBagConstraints);

        titleLabel1.setText("contains images -- in subdirectories or directly. E.g., if the root of the \"Sao Tome\" series \nis the");
        titleLabel1.setMaximumSize(new java.awt.Dimension(530, 17));
        titleLabel1.setMinimumSize(new java.awt.Dimension(530, 17));
        titleLabel1.setPreferredSize(new java.awt.Dimension(530, 17));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 6);
        listPanel.add(titleLabel1, gridBagConstraints);

        titleLabel2.setText("directory \"C:\\My Photos\\Sao-Tome\", the series container directory is \"C:\\My Photos\".");
        titleLabel2.setMaximumSize(new java.awt.Dimension(478, 17));
        titleLabel2.setMinimumSize(new java.awt.Dimension(478, 17));
        titleLabel2.setPreferredSize(new java.awt.Dimension(478, 17));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 6);
        listPanel.add(titleLabel2, gridBagConstraints);

        sContainerDirList.setVisibleRowCount(6);
        jScrollPane1.setViewportView(sContainerDirList);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridheight = 5;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.ipadx = 4;
        gridBagConstraints.ipady = 4;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.weighty = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(4, 6, 6, 4);
        listPanel.add(jScrollPane1, gridBagConstraints);

        addButton.setText("Add ...");
        addButton.setToolTipText("Add directory containing the series directories");
        addButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                SettingsDialog.this.addButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
        gridBagConstraints.insets = new java.awt.Insets(3, 0, 0, 6);
        listPanel.add(addButton, gridBagConstraints);

        delButton.setText("Remove");
        delButton.setToolTipText("Removes the selected directory");
        delButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                SettingsDialog.this.delButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
        gridBagConstraints.insets = new java.awt.Insets(3, 0, 0, 6);
        listPanel.add(delButton, gridBagConstraints);

        moveUpButton.setText("Move Up");
        moveUpButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                SettingsDialog.this.moveUpButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(10, 0, 0, 6);
        listPanel.add(moveUpButton, gridBagConstraints);

        moveDownButton.setText("Move Down");
        moveDownButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                SettingsDialog.this.moveDownButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 7;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
        gridBagConstraints.insets = new java.awt.Insets(3, 0, 6, 6);
        listPanel.add(moveDownButton, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 6);
        seriesPanel.add(listPanel, gridBagConstraints);

        sTitlePanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Series Structure"));
        sTitlePanel.setOpaque(false);
        sTitlePanel.setLayout(new java.awt.GridBagLayout());

        jLabel4.setText("Title:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 0);
        sTitlePanel.add(jLabel4, gridBagConstraints);

        sTitleText.setColumns(50);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 6);
        sTitlePanel.add(sTitleText, gridBagConstraints);

        jLabel5.setText("Description:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(3, 6, 0, 4);
        sTitlePanel.add(jLabel5, gridBagConstraints);

        sDescTextArea.setColumns(66);
        sDescTextArea.setLineWrap(true);
        sDescTextArea.setRows(3);
        sDescTextArea.setWrapStyleWord(true);
        jScrollPane2.setViewportView(sDescTextArea);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.weighty = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 6, 6);
        sTitlePanel.add(jScrollPane2, gridBagConstraints);

        jLabel1.setText("Image Copyright (${year} stands for the year):");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        sTitlePanel.add(jLabel1, gridBagConstraints);

        sCopyrightText.setText(" ${year}");
        sCopyrightText.setToolTipText("Copyright text to be printed on every exported image (original images are unaffected). Leave empty for no copyright text.");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        sTitlePanel.add(sCopyrightText, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.weighty = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 6, 6);
        seriesPanel.add(sTitlePanel, gridBagConstraints);

        jTabbedPane.addTab("Series", seriesPanel);

        exportPanel.setOpaque(false);
        exportPanel.setLayout(new java.awt.GridBagLayout());

        exportDirPanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Web Export Directory"));
        exportDirPanel.setOpaque(false);
        exportDirPanel.setLayout(new java.awt.GridBagLayout());

        exportDirLabel.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
        exportDirLabel.setText("Export means that the images are prepared to be put on the Web: they are \nscaled and converted");
        exportDirLabel.setMaximumSize(new java.awt.Dimension(543, 17));
        exportDirLabel.setMinimumSize(new java.awt.Dimension(543, 17));
        exportDirLabel.setPreferredSize(new java.awt.Dimension(543, 17));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 7;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 6);
        exportDirPanel.add(exportDirLabel, gridBagConstraints);

        exportDirLabel1.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
        exportDirLabel1.setText("to JPEG. The original images remain unaffected, exporting always creates image copies. The");
        exportDirLabel1.setMaximumSize(new java.awt.Dimension(524, 17));
        exportDirLabel1.setMinimumSize(new java.awt.Dimension(524, 17));
        exportDirLabel1.setPreferredSize(new java.awt.Dimension(524, 17));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 7;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 6);
        exportDirPanel.add(exportDirLabel1, gridBagConstraints);

        exportDirLabel2.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
        exportDirLabel2.setText("\"export directory\" is the directory containing copies of the series, where the images are Web-");
        exportDirLabel2.setMaximumSize(new java.awt.Dimension(531, 17));
        exportDirLabel2.setMinimumSize(new java.awt.Dimension(531, 17));
        exportDirLabel2.setPreferredSize(new java.awt.Dimension(531, 17));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 7;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 6);
        exportDirPanel.add(exportDirLabel2, gridBagConstraints);

        exportDirLabel3.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
        exportDirLabel3.setText("ready. E.g., if the Web images of the \"Sao Tome\" series should be in \n\"C:\\My Web\\phto\\sao-tome\"");
        exportDirLabel3.setMaximumSize(new java.awt.Dimension(542, 17));
        exportDirLabel3.setMinimumSize(new java.awt.Dimension(542, 17));
        exportDirLabel3.setPreferredSize(new java.awt.Dimension(542, 17));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 7;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 6);
        exportDirPanel.add(exportDirLabel3, gridBagConstraints);

        exportDirLabel4.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
        exportDirLabel4.setText("the export directory is \"C:\\My Web\\phto\".");
        exportDirLabel4.setMaximumSize(new java.awt.Dimension(230, 17));
        exportDirLabel4.setMinimumSize(new java.awt.Dimension(230, 17));
        exportDirLabel4.setPreferredSize(new java.awt.Dimension(230, 17));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.gridwidth = 7;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 6);
        exportDirPanel.add(exportDirLabel4, gridBagConstraints);

        exportDirLabel5.setText("Web Export Directory:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 0);
        exportDirPanel.add(exportDirLabel5, gridBagConstraints);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.gridwidth = 5;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 0);
        exportDirPanel.add(exportDirText, gridBagConstraints);

        exportDirSelButton.setText("...");
        exportDirSelButton.setToolTipText("Select Web export directory");
        exportDirSelButton.setMaximumSize(new java.awt.Dimension(30, 20));
        exportDirSelButton.setMinimumSize(new java.awt.Dimension(30, 20));
        exportDirSelButton.setPreferredSize(new java.awt.Dimension(30, 20));
        exportDirSelButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                SettingsDialog.this.exportDirSelButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 6;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.insets = new java.awt.Insets(6, 4, 6, 6);
        exportDirPanel.add(exportDirSelButton, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 6);
        exportPanel.add(exportDirPanel, gridBagConstraints);

        convSizePanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Image Conversion"));
        convSizePanel.setOpaque(false);
        convSizePanel.setLayout(new java.awt.GridBagLayout());

        mainImageLabel.setText("Main Images (each to be shown on its own Web page):");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 6;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 0);
        convSizePanel.add(mainImageLabel, gridBagConstraints);

        mainLengthLabel.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        mainLengthLabel.setText("Length:");
        mainLengthLabel.setMaximumSize(new java.awt.Dimension(110, 16));
        mainLengthLabel.setMinimumSize(new java.awt.Dimension(110, 16));
        mainLengthLabel.setOpaque(false);
        mainLengthLabel.setPreferredSize(new java.awt.Dimension(110, 16));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 0, 0);
        convSizePanel.add(mainLengthLabel, gridBagConstraints);

        mainLengthText.setToolTipText("Main image normal width or height in pixels");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 4, 0, 0);
        convSizePanel.add(mainLengthText, gridBagConstraints);

        mainMaxWidthLabel.setText("Max. Width:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.insets = new java.awt.Insets(6, 12, 0, 0);
        convSizePanel.add(mainMaxWidthLabel, gridBagConstraints);

        mainMaxWidthText.setToolTipText("Maximum main image width in pixels");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 4, 0, 0);
        convSizePanel.add(mainMaxWidthText, gridBagConstraints);

        mainMaxHeightLabel.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        mainMaxHeightLabel.setText("Max. Height:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(6, 12, 0, 0);
        convSizePanel.add(mainMaxHeightLabel, gridBagConstraints);

        mainMaxHeightText.setToolTipText("Maximum main image height in pixels");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 4, 0, 6);
        convSizePanel.add(mainMaxHeightText, gridBagConstraints);

        mainQualityLabel.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        mainQualityLabel.setText("JPEG Quality:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 0, 0);
        convSizePanel.add(mainQualityLabel, gridBagConstraints);

        mainQualityText.setToolTipText("Main image JPEG compression quality in percent");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 4, 0, 0);
        convSizePanel.add(mainQualityText, gridBagConstraints);

        percentLabel.setText("%");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(6, 3, 0, 0);
        convSizePanel.add(percentLabel, gridBagConstraints);

        jSeparator2.setForeground(new java.awt.Color(208, 208, 191));
        jSeparator2.setMinimumSize(new java.awt.Dimension(10, 2));
        jSeparator2.setPreferredSize(new java.awt.Dimension(10, 2));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 6;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(8, 6, 0, 6);
        convSizePanel.add(jSeparator2, gridBagConstraints);

        indexImageLabel.setText("Overview Images (all of a group to be shown together on an overview [index] page):");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.gridwidth = 6;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 0, 0);
        convSizePanel.add(indexImageLabel, gridBagConstraints);

        indexLengthLabel.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        indexLengthLabel.setText("Max. Size:");
        indexLengthLabel.setMaximumSize(new java.awt.Dimension(110, 16));
        indexLengthLabel.setMinimumSize(new java.awt.Dimension(110, 16));
        indexLengthLabel.setPreferredSize(new java.awt.Dimension(110, 16));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 0, 0);
        convSizePanel.add(indexLengthLabel, gridBagConstraints);

        indexLengthText.setToolTipText("Normal overview image width in pixels");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 4, 0, 0);
        convSizePanel.add(indexLengthText, gridBagConstraints);

        indexQualityLabel.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        indexQualityLabel.setText("JPEG Quality:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 0);
        convSizePanel.add(indexQualityLabel, gridBagConstraints);

        indexQualityText.setToolTipText("Overview image JPEG compression quality in percent");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 4, 6, 0);
        convSizePanel.add(indexQualityText, gridBagConstraints);

        percentLabel1.setText("%");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(6, 3, 0, 0);
        convSizePanel.add(percentLabel1, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 6, 6);
        exportPanel.add(convSizePanel, gridBagConstraints);

        convToolPanel1.setBorder(javax.swing.BorderFactory.createTitledBorder("Conversion Tool"));
        convToolPanel1.setOpaque(false);
        convToolPanel1.setLayout(new java.awt.GridBagLayout());

        useMagickCheckBox.setText("Use ImageMagick");
        useMagickCheckBox.setOpaque(false);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 6);
        convToolPanel1.add(useMagickCheckBox, gridBagConstraints);

        useMagickDescLabel.setText(" ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 0, 6, 6);
        convToolPanel1.add(useMagickDescLabel, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 6, 6);
        exportPanel.add(convToolPanel1, gridBagConstraints);

        jTabbedPane.addTab("Web Export", exportPanel);

        savePanel.setOpaque(false);
        savePanel.setLayout(new java.awt.GridBagLayout());

        metadataFormatPanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Metadata Format"));
        metadataFormatPanel.setOpaque(false);
        metadataFormatPanel.setLayout(new java.awt.GridBagLayout());

        saveLabel.setForeground(new java.awt.Color(0, 70, 213));
        saveLabel.setText("Series Metadata");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(3, 6, 0, 0);
        metadataFormatPanel.add(saveLabel, gridBagConstraints);

        saveTitleLabel.setText("Check to save the series metadata as XML, uncheck to save as text.");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(3, 6, 0, 0);
        metadataFormatPanel.add(saveTitleLabel, gridBagConstraints);

        saveAsXMLCheckBox.setText("Save as XML");
        saveAsXMLCheckBox.setToolTipText("Check to save the series metadata as XML, uncheck to save as text");
        saveAsXMLCheckBox.setBorder(javax.swing.BorderFactory.createEmptyBorder(0, 0, 0, 0));
        saveAsXMLCheckBox.setMargin(new java.awt.Insets(0, 0, 0, 0));
        saveAsXMLCheckBox.setOpaque(false);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(3, 8, 0, 0);
        metadataFormatPanel.add(saveAsXMLCheckBox, gridBagConstraints);

        saveEncodingLabel.setText("Character Encoding:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(3, 3, 0, 0);
        metadataFormatPanel.add(saveEncodingLabel, gridBagConstraints);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(3, 6, 0, 6);
        metadataFormatPanel.add(saveEncodingTextField, gridBagConstraints);

        jSeparator3.setForeground(new java.awt.Color(208, 208, 191));
        jSeparator3.setMinimumSize(new java.awt.Dimension(10, 2));
        jSeparator3.setPreferredSize(new java.awt.Dimension(10, 2));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(12, 6, 0, 8);
        metadataFormatPanel.add(jSeparator3, gridBagConstraints);

        exportLabel.setForeground(new java.awt.Color(0, 70, 213));
        exportLabel.setText("Web-Exported Metadata");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(9, 6, 0, 0);
        metadataFormatPanel.add(exportLabel, gridBagConstraints);

        exportTitleLabel.setText("Check to export the series metadata as XML, uncheck to export as text.");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(3, 6, 0, 0);
        metadataFormatPanel.add(exportTitleLabel, gridBagConstraints);

        exportAsXmlCheckBox.setText("Export as XML");
        exportAsXmlCheckBox.setToolTipText("Check to export the series metadata as XML, uncheck to export as text");
        exportAsXmlCheckBox.setBorder(javax.swing.BorderFactory.createEmptyBorder(0, 0, 0, 0));
        exportAsXmlCheckBox.setMargin(new java.awt.Insets(0, 0, 0, 0));
        exportAsXmlCheckBox.setOpaque(false);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(3, 8, 0, 0);
        metadataFormatPanel.add(exportAsXmlCheckBox, gridBagConstraints);

        exportEncodingLabel.setText("Character Encoding:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(3, 3, 0, 0);
        metadataFormatPanel.add(exportEncodingLabel, gridBagConstraints);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(3, 6, 0, 6);
        metadataFormatPanel.add(exportEncodingTextField, gridBagConstraints);

        jSeparator4.setForeground(new java.awt.Color(208, 208, 191));
        jSeparator4.setMinimumSize(new java.awt.Dimension(10, 2));
        jSeparator4.setPreferredSize(new java.awt.Dimension(10, 2));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 7;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(12, 6, 0, 8);
        metadataFormatPanel.add(jSeparator4, gridBagConstraints);

        restartLabel.setForeground(new java.awt.Color(0, 70, 213));
        restartLabel.setText("These settings take effect after a restart of JPhoto-Explorer.");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 8;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(10, 6, 0, 0);
        metadataFormatPanel.add(restartLabel, gridBagConstraints);

        restartLabel2.setText("Consider backing up your series metadata files before restarting.");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 9;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 6, 4, 0);
        metadataFormatPanel.add(restartLabel2, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.weighty = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 0, 6);
        savePanel.add(metadataFormatPanel, gridBagConstraints);

        jTabbedPane.addTab("Save", savePanel);

        panelOptionsPanel.setOpaque(false);
        panelOptionsPanel.setLayout(new java.awt.GridBagLayout());

        dirTreePanelOptionsPanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Directory Tree Panel"));
        dirTreePanelOptionsPanel.setOpaque(false);
        dirTreePanelOptionsPanel.setLayout(new java.awt.GridBagLayout());

        saveCurrentDirCheckBox.setText("Restore current directory selection on startup");
        saveCurrentDirCheckBox.setBorder(javax.swing.BorderFactory.createEmptyBorder(0, 0, 0, 0));
        saveCurrentDirCheckBox.setMargin(new java.awt.Insets(0, 0, 0, 0));
        saveCurrentDirCheckBox.setOpaque(false);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 6, 3, 0);
        dirTreePanelOptionsPanel.add(saveCurrentDirCheckBox, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 0, 6);
        panelOptionsPanel.add(dirTreePanelOptionsPanel, gridBagConstraints);

        contentsPanelOptionsPanel.setBorder(javax.swing.BorderFactory.createTitledBorder(" Contents Panel"));
        contentsPanelOptionsPanel.setOpaque(false);
        contentsPanelOptionsPanel.setLayout(new java.awt.GridBagLayout());

        doubleClickOpensDirCheckBox.setText("Double click on directory opens");
        doubleClickOpensDirCheckBox.setBorder(javax.swing.BorderFactory.createEmptyBorder(0, 0, 0, 0));
        doubleClickOpensDirCheckBox.setMargin(new java.awt.Insets(0, 0, 0, 0));
        doubleClickOpensDirCheckBox.setOpaque(false);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(3, 6, 0, 0);
        contentsPanelOptionsPanel.add(doubleClickOpensDirCheckBox, gridBagConstraints);

        doubleClickOpensFileCheckBox.setText("Double click on file opens");
        doubleClickOpensFileCheckBox.setBorder(javax.swing.BorderFactory.createEmptyBorder(0, 0, 0, 0));
        doubleClickOpensFileCheckBox.setMargin(new java.awt.Insets(0, 0, 0, 0));
        doubleClickOpensFileCheckBox.setOpaque(false);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 4, 0);
        contentsPanelOptionsPanel.add(doubleClickOpensFileCheckBox, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.weighty = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 6);
        panelOptionsPanel.add(contentsPanelOptionsPanel, gridBagConstraints);

        jTabbedPane.addTab("Panel Options", panelOptionsPanel);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 6);
        this.getContentPane().add(jTabbedPane, gridBagConstraints);

        buttonPanel.setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.RIGHT));

        okButton.setText("OK");
        okButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                SettingsDialog.this.okButtonActionPerformed(evt);
            }
        });
        buttonPanel.add(okButton);

        cancelButton.setText("Cancel");
        cancelButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                SettingsDialog.this.cancelButtonActionPerformed(evt);
            }
        });
        buttonPanel.add(cancelButton);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 3, 6);
        this.getContentPane().add(buttonPanel, gridBagConstraints);

        this.pack();
    }// </editor-fold>//GEN-END:initComponents

    private void exportDirSelButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_exportDirSelButtonActionPerformed
    {//GEN-HEADEREND:event_exportDirSelButtonActionPerformed
        JFileChooser choose = new JFileChooser();
        File         dirFile;
        String       dir;

        // Prepare the file chooser dialog.
        choose.setDialogTitle("Select Web Export Directory");
        choose.setDialogType(JFileChooser.OPEN_DIALOG);
        choose.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        choose.setApproveButtonToolTipText(
            "Select as root directory for the Web export of all series");
        dir = exportDirText.getText();
        if (dir != null && dir.length() > 0)
        {
            dirFile = new File(dir);
            dirFile = dirFile.getParentFile();
            if (dirFile.isDirectory())
                choose.setCurrentDirectory(dirFile);
        }
        // Show the file chooser dialog.
        if (choose.showDialog(this,"Select") != JFileChooser.APPROVE_OPTION)
            return;
        // Get the selected directory.
        dirFile = choose.getSelectedFile();
        if (dirFile == null || !dirFile.isDirectory())
            return;
        dir = dirFile.getAbsolutePath();
        exportDirText.setText(dir);
    }//GEN-LAST:event_exportDirSelButtonActionPerformed

    private void delButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_delButtonActionPerformed
    {//GEN-HEADEREND:event_delButtonActionPerformed
        int index = sContainerDirList.getSelectedIndex();

        if (index < 0 || index >= m_seriesContainerNames.size())
            return;
        // Remove the current series container directory.
        m_seriesContainerNames.removeElementAt(index);
        sContainerDirList.setListData(m_seriesContainerNames);
        m_currentSContainerDir = null;
    }//GEN-LAST:event_delButtonActionPerformed

    private void moveDownButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_moveDownButtonActionPerformed
    {//GEN-HEADEREND:event_moveDownButtonActionPerformed
        int index = sContainerDirList.getSelectedIndex();

        if (index < 0 || index >= m_seriesContainerNames.size() - 1)
            return;
        String th = m_seriesContainerNames.get(index);
        String nx = m_seriesContainerNames.get(index+1);
        m_seriesContainerNames.setElementAt(th, index+1);
        m_seriesContainerNames.setElementAt(nx, index);
        sContainerDirList.setListData(m_seriesContainerNames);
        sContainerDirList.setSelectedIndex(index+1);
        m_currentSContainerDir = null;
    }//GEN-LAST:event_moveDownButtonActionPerformed

    private void moveUpButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_moveUpButtonActionPerformed
    {//GEN-HEADEREND:event_moveUpButtonActionPerformed
        int index = sContainerDirList.getSelectedIndex();

        if (index <= 0 || index >= m_seriesContainerNames.size())
            return;
        String th = m_seriesContainerNames.get(index);
        String nx = m_seriesContainerNames.get(index-1);
        m_seriesContainerNames.setElementAt(th, index-1);
        m_seriesContainerNames.setElementAt(nx, index);
        sContainerDirList.setListData(m_seriesContainerNames);
        sContainerDirList.setSelectedIndex(index-1);
        m_currentSContainerDir = null;
    }//GEN-LAST:event_moveUpButtonActionPerformed

    /**
     * Handler for approving the settings and finishing the dialog.
     */
    private void okButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_okButtonActionPerformed
    {//GEN-HEADEREND:event_okButtonActionPerformed
        File    f;
        String  s, exportDir;
        Integer intVal;
        int     mainLength, mainMaxWidth, mainMaxHeight, mainQuality;
        int     indexLength, indexQuality;

        // Validate the series containers.
        if (m_seriesContainerNames.size() == 0)
        {
            JOptionPane.showMessageDialog(this, "Please select at least one "
                + "series container directory", AppInfo.APP_NAME,
                JOptionPane.WARNING_MESSAGE);
            // Select the offending component.
            jTabbedPane.setSelectedIndex(0);
            addButton.requestFocus();
            return;
        }
        // Validate the web export directory.
        exportDir = exportDirText.getText();
        f = new File(exportDir);
        if (!f.isDirectory())
        {
            JOptionPane.showMessageDialog(this, "Please enter or select a "
                + "valid Web export directory", AppInfo.APP_NAME,
                JOptionPane.WARNING_MESSAGE);
            // Select the offending component.
            jTabbedPane.setSelectedIndex(1);
            exportDirText.requestFocus();
            return;
        }
        // Validate the image size properties.
        intVal = this.validateInt(mainLengthText, "normal width/height", 1, 0, 10000);
        if (intVal == null)
        	return;
        mainLength = intVal.intValue();
        intVal = this.validateInt(mainMaxWidthText, "maximum width", 1,
            0, 10000);
        if (intVal == null)
            return;
        mainMaxWidth = intVal.intValue();
        intVal = this.validateInt(mainMaxHeightText, "maximum height", 1,
            0, 10000);
        if (intVal == null)
            return;
        mainMaxHeight = intVal.intValue();
        intVal = this.validateInt(mainQualityText, "JPEG quality", 1, 0, 100);
        if (intVal == null)
            return;
        mainQuality = intVal.intValue();
        intVal = this.validateInt(indexLengthText, "maximum width/height", 1, 0, 10000);
        if (intVal == null)
            return;
        indexLength = intVal.intValue();
        intVal = this.validateInt(indexQualityText, "JPEG quality", 1, 0, 100);
        if (intVal == null)
            return;
        indexQuality = intVal.intValue();
        // Let all panels save their settings before the settings are updated.
        m_frame.saveSettings();
        // Save the changes into the settings object.
        s = sTitleText.getText();
        if (s.length() > 0)
            m_settings.set(Settings.GALLERY_TITLE, s);
        s = sDescTextArea.getText();
        if (s.length() > 0)
            m_settings.set(Settings.GALLERY_DESCRIPTION, s);
        s = sCopyrightText.getText();
        if (s.length() > 0 && !Settings.DEFAULT_COPYRIGHT_STRING.equals(s))
            m_settings.set(Settings.GALLERY_COPYRIGHT, s);
        m_settings.set(Settings.EXPORT_DIRECTORY, exportDir);
        m_settings.setInt(Settings.EXPORT_MAINIMG_LENGTH, mainLength);
        m_settings.setInt(Settings.EXPORT_MAINIMG_MAX_WIDTH, mainMaxWidth);
        m_settings.setInt(Settings.EXPORT_MAINIMG_MAX_HEIGHT, mainMaxHeight);
        m_settings.setInt(Settings.EXPORT_MAINIMG_QUALITY, mainQuality);
        m_settings.setInt(Settings.EXPORT_INDEXIMG_LENGTH, indexLength);
        m_settings.setInt(Settings.EXPORT_INDEXIMG_QUALITY, indexQuality);
        // Set whether we are using ImageMagick (only if we have the choice).
        if (m_bHaveImageMagick)
        {
            m_settings.setInt(Settings.EXPORT_USE_IMAGE_MAGICK,
                useMagickCheckBox.isSelected() ? 1 : 0);
        }
        // Set the metadata format options.
        m_settings.setBoolean(Settings.METADATA_SAVE_XML,
            saveAsXMLCheckBox.isSelected());
        s = saveEncodingTextField.getText();
        if (s != null && s.length() > 0 &&
            !s.equalsIgnoreCase(Settings.METADEFAULT_XML_ENCODING))
        {
            m_settings.set(Settings.METADATA_SAVE_ENCODING, s);
        }
        else
            m_settings.remove(Settings.METADATA_SAVE_ENCODING);
        m_settings.setBoolean(Settings.METADATA_EXPORT_XML,
            exportAsXmlCheckBox.isSelected());
        s = exportEncodingTextField.getText();
        if (s != null && s.length() > 0 &&
            !s.equalsIgnoreCase(Settings.METADEFAULT_XML_ENCODING))
        {
            m_settings.set(Settings.METADATA_EXPORT_ENCODING, s);
        }
        else
            m_settings.remove(Settings.METADATA_EXPORT_ENCODING);
        // Now we set the panel options.
        // For the directory tree panel.
        m_settings.setBoolean(DirTreePanel.SAVE_CURRENT_DIR_ON_EXIT,
            saveCurrentDirCheckBox.isSelected());
        // For the directory contents panel.
        m_settings.setBoolean(ContentsPanel.DBL_CLICK_DIR_OPENS,
            doubleClickOpensDirCheckBox.isSelected());
        m_settings.setBoolean(ContentsPanel.DBL_CLICK_FILE_OPENS,
            doubleClickOpensFileCheckBox.isSelected());
        // At last, we set the series directories and save the settings object.
        try
        {
            m_seriesContainer.mergeSeriesContainers(m_seriesContainerNames);
            Iterator<String> iter = m_seriesContainerNames.iterator();
            while (iter.hasNext())
                m_settings.addSeriesContainer(iter.next());
            m_settings.save();
        }
        catch (Exception pe)
        {
            // Show the save settings error dialog.
            JOptionPane.showMessageDialog(this, pe.getMessage(),
                AppInfo.APP_NAME, JOptionPane.ERROR_MESSAGE);
            return;
        }
        // Now let all reload their settings.
        m_frame.reloadSettings();
        // Close this dialog window.
        m_isChanged = true;
        this.cancelButtonActionPerformed(evt);
    }//GEN-LAST:event_okButtonActionPerformed

    private void addButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_addButtonActionPerformed
    {//GEN-HEADEREND:event_addButtonActionPerformed
        JFileChooser choose = new JFileChooser();
        File         dirFile;
        String       dir;

        // Prepare the file chooser dialog.
        choose.setDialogTitle("Select Series Container Directory");
        choose.setDialogType(JFileChooser.OPEN_DIALOG);
        choose.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        choose.setApproveButtonToolTipText(
            "Select as Series Container directory");
        dir = m_currentSContainerDir;
        if (dir == null && m_seriesContainerNames.size() > 0)
            dir = m_seriesContainerNames.get(0);
        if (dir != null && dir.length() > 0)
        {
            dirFile = new File(dir);
            if (dirFile.isDirectory())
                choose.setCurrentDirectory(dirFile);
        }
        // Show the file chooser dialog.
        if (choose.showDialog(this,"Select") != JFileChooser.APPROVE_OPTION)
            return;
        // Get the selected directory.
        dirFile = choose.getSelectedFile();
        if (dirFile == null || !dirFile.isDirectory())
            return;
        dir = dirFile.getAbsolutePath();
        if (!m_seriesContainerNames.contains(dir))
        {
            m_seriesContainerNames.add(dir);
            sContainerDirList.setListData(m_seriesContainerNames);
        }
    }//GEN-LAST:event_addButtonActionPerformed

    private void cancelButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_cancelButtonActionPerformed
    {//GEN-HEADEREND:event_cancelButtonActionPerformed
        this.setVisible(false);
        this.dispose();
    }//GEN-LAST:event_cancelButtonActionPerformed

    /** Closes the dialog */
    private void closeDialog(java.awt.event.WindowEvent evt)
    {//GEN-FIRST:event_closeDialog
        this.setVisible(false);
        this.dispose();
    }//GEN-LAST:event_closeDialog

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton addButton;
    private javax.swing.JPanel buttonPanel;
    private javax.swing.JButton cancelButton;
    private javax.swing.JPanel contentsPanelOptionsPanel;
    private javax.swing.JPanel convSizePanel;
    private javax.swing.JPanel convToolPanel1;
    private javax.swing.JButton delButton;
    private javax.swing.JPanel dirTreePanelOptionsPanel;
    private javax.swing.JCheckBox doubleClickOpensDirCheckBox;
    private javax.swing.JCheckBox doubleClickOpensFileCheckBox;
    private javax.swing.JCheckBox exportAsXmlCheckBox;
    private javax.swing.JLabel exportDirLabel;
    private javax.swing.JLabel exportDirLabel1;
    private javax.swing.JLabel exportDirLabel2;
    private javax.swing.JLabel exportDirLabel3;
    private javax.swing.JLabel exportDirLabel4;
    private javax.swing.JLabel exportDirLabel5;
    private javax.swing.JPanel exportDirPanel;
    private javax.swing.JButton exportDirSelButton;
    private javax.swing.JTextField exportDirText;
    private javax.swing.JLabel exportEncodingLabel;
    private javax.swing.JTextField exportEncodingTextField;
    private javax.swing.JLabel exportLabel;
    private javax.swing.JPanel exportPanel;
    private javax.swing.JLabel exportTitleLabel;
    private javax.swing.JLabel indexImageLabel;
    private javax.swing.JLabel indexLengthLabel;
    private javax.swing.JTextField indexLengthText;
    private javax.swing.JLabel indexQualityLabel;
    private javax.swing.JTextField indexQualityText;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JLabel jLabel5;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JScrollPane jScrollPane2;
    private javax.swing.JSeparator jSeparator2;
    private javax.swing.JSeparator jSeparator3;
    private javax.swing.JSeparator jSeparator4;
    private javax.swing.JTabbedPane jTabbedPane;
    private javax.swing.JPanel listPanel;
    private javax.swing.JLabel mainImageLabel;
    private javax.swing.JLabel mainMaxWidthLabel;
    private javax.swing.JTextField mainMaxWidthText;
    private javax.swing.JLabel mainMaxHeightLabel;
    private javax.swing.JTextField mainMaxHeightText;
    private javax.swing.JLabel mainQualityLabel;
    private javax.swing.JTextField mainQualityText;
    private javax.swing.JLabel mainLengthLabel;
    private javax.swing.JTextField mainLengthText;
    private javax.swing.JPanel metadataFormatPanel;
    private javax.swing.JButton moveDownButton;
    private javax.swing.JButton moveUpButton;
    private javax.swing.JButton okButton;
    private javax.swing.JPanel panelOptionsPanel;
    private javax.swing.JLabel percentLabel;
    private javax.swing.JLabel percentLabel1;
    private javax.swing.JLabel restartLabel;
    private javax.swing.JLabel restartLabel2;
    private javax.swing.JList sContainerDirList;
    private javax.swing.JTextField sCopyrightText;
    private javax.swing.JTextArea sDescTextArea;
    private javax.swing.JPanel sTitlePanel;
    private javax.swing.JTextField sTitleText;
    private javax.swing.JCheckBox saveAsXMLCheckBox;
    private javax.swing.JCheckBox saveCurrentDirCheckBox;
    private javax.swing.JLabel saveEncodingLabel;
    private javax.swing.JTextField saveEncodingTextField;
    private javax.swing.JLabel saveLabel;
    private javax.swing.JPanel savePanel;
    private javax.swing.JLabel saveTitleLabel;
    private javax.swing.JPanel seriesPanel;
    private javax.swing.JLabel titleLabel;
    private javax.swing.JLabel titleLabel1;
    private javax.swing.JLabel titleLabel2;
    private javax.swing.JCheckBox useMagickCheckBox;
    private javax.swing.JLabel useMagickDescLabel;
    // End of variables declaration//GEN-END:variables

} // SettingsDialog
