/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: DirTreePanel
    Created: 17.12.2005 (19:00:41)
        $Id: DirTreePanel.java 160 2009-05-31 07:57:29Z dirk $
  $Revision: 160 $
      $Date: 2009-05-31 09:57:29 +0200 (So, 31 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui.panel;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JTree;
import javax.swing.event.TreeExpansionEvent;
import javax.swing.event.TreeExpansionListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import com.dgrossmann.photo.AppInfo;
import com.dgrossmann.photo.dir.AbstractFSObject;
import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.persist.PersistException;
import com.dgrossmann.photo.settings.Settings;
import com.dgrossmann.photo.ui.ExplorerMainFrame;

/**
 * Panel class for the directory tree panel.
 * @author Dirk Grossmann
 */
public class DirTreePanel extends AbstractExplorerPanel
    implements TreeSelectionListener, TreeExpansionListener,
    ActionListener, FocusListener
{
    public static final String SAVE_CURRENT_DIR_ON_EXIT =
        "dirtree.save_current_dir_on_exit";
    public static final String CURRENT_DIR =
        "dirtree.current_dir";

    private static final String DIR_PANE_TITLE = "Series";

    private JTree        m_dirTree;
    private DirTreeModel m_dirTreeModel;
    private boolean      m_bSaveCurrentDirOnExit;

    // Directory popup menu and items.
    private JPopupMenu   m_dirPopupMenu;
    private JMenuItem    m_dpNewDir, m_dpPaste;
    private JMenuItem    m_dpProperties, m_dpCut, m_dpDelete, m_dpExport;

    /**
     * Creates a new <tt>DirTreePanel</tt> instance.
     * @param frm - The parent explorer frame
     */
    public DirTreePanel (ExplorerMainFrame frm)
    {
        super(frm, DIR_PANE_TITLE);
        m_dirPopupMenu = null;
        m_bSaveCurrentDirOnExit = false;
        // Initialize the component.
        this.setLayout(new BorderLayout());
        this.add(this.getTitleLabel(), BorderLayout.NORTH);
        // Add the directory tree component.
        m_dirTreeModel = new DirTreeModel(this.getFrame().getSeriesContainer(),
            this.getFrame());
        m_dirTree = new JTree(m_dirTreeModel);
        m_dirTree.setVisibleRowCount(80);
        m_dirTree.setEditable(true);
        m_dirTree.putClientProperty("JTree.lineStyle", "Angled");
        m_dirTree.getSelectionModel().setSelectionMode
            (TreeSelectionModel.SINGLE_TREE_SELECTION);
        m_dirTree.addTreeSelectionListener(this);
        m_dirTree.addTreeExpansionListener(this);
        m_dirTree.addMouseListener(new DirTreeMouseListener(this));
        m_dirTree.addFocusListener(this);
        this.add(new JScrollPane(m_dirTree,
            JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
            JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED),
            BorderLayout.CENTER);
        // Expand and select the root node.
        if (m_dirTree.getRowCount() > 0)
        {
            m_dirTree.expandRow(0);
            m_dirTree.setSelectionRow(0);
        }
    } // DirTreePanel

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#setupComponents()
     */
    public void setupComponents ()
    {
    } // setupComponents

    /**
     * @see com.dgrossmann.photo.ui.panel.AbstractExplorerPanel#setCurrentDirectory(com.dgrossmann.photo.dir.DirectoryObject)
     */
    @Override
	public void setCurrentDirectory (DirectoryObject currentDirectory)
    {
        DirectoryObject oldDir = this.getCurrentDirectory();
        super.setCurrentDirectory(currentDirectory);
        // Optimize for the usual case that the new directory is an immediate
        // child of the old one.
        if (oldDir.getIndexOfSubDir(currentDirectory) >= 0)
        {
            TreePath selPath = m_dirTree.getSelectionPath();
            m_dirTree.expandPath(selPath);
            Object[] oldPath = selPath.getPath();
            Object[] newPath = new Object[oldPath.length + 1];
            int i;
            for (i = 0; i < oldPath.length; i++)
                newPath[i] = oldPath[i];
            newPath[i] = currentDirectory;
            m_dirTree.setSelectionPath(new TreePath(newPath));
            return;
        }
    } // setCurrentDirectory

    /**
     * @see javax.swing.event.TreeExpansionListener#treeExpanded(javax.swing.event.TreeExpansionEvent)
     */
    public void treeExpanded (TreeExpansionEvent event)
    {
        Object node = event.getPath().getLastPathComponent();
        if (node == null || !(node instanceof DirectoryObject))
            return;
        DirectoryObject dir = (DirectoryObject) node;
        if (dir.getParent() == null)
        {
            this.setCursor(new Cursor(Cursor.WAIT_CURSOR));
            this.getFrame().getSeriesContainer().ensureLoaded(dir);
            m_dirTreeModel.fireTreeStructureChanged(dir);
            this.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
        }
    } // treeExpanded

    /**
     * @see javax.swing.event.TreeExpansionListener#treeCollapsed(javax.swing.event.TreeExpansionEvent)
     */
    public void treeCollapsed (TreeExpansionEvent event)
    {
    } // treeCollapsed

    /**
     * @see javax.swing.event.TreeSelectionListener#valueChanged(javax.swing.event.TreeSelectionEvent)
     */
    public void valueChanged (TreeSelectionEvent e)
    {
        this.getFrame().save();
        Object node = m_dirTree.getLastSelectedPathComponent();
        // Check for the root (the misused file).
        if (node == null || !(node instanceof DirectoryObject))
        {
            // We call our base class' method as we try to expand the tree.
            super.setCurrentDirectory(null);
            this.fireCurrentDirectoryChanged(null);
        }
        else
        {
            // Change the current directory entry.
            DirectoryObject currentDir = (DirectoryObject) node;
            // We call our base class' method as we try to expand the tree.
            super.setCurrentDirectory(currentDir);
            // Ensure that the series is loaded.
            if (currentDir.getParent() == null)
            {
                this.setCursor(new Cursor(Cursor.WAIT_CURSOR));
                this.getFrame().getSeriesContainer().ensureLoaded(currentDir);
                this.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
            }
            this.fireCurrentDirectoryChanged(currentDir);
        }
    } // valueChanged

    /**
     * Tries to select the directory node denoted by a tree path.
     * @param treePath - Tree path of the directory to be selected
     * @param bExact - <tt>True</tt> if an exact match is needed, <tt>false</tt>
     * if an ancestor of the node denoted by the tree path is sufficient
     * @return The actually selected directory object, or <tt>null</tt>
     * otherwise
     */
    public DirectoryObject selectDirForPath
        ( TreePath treePath
        , boolean  bExact
        )
    {
        if (treePath == null || m_dirTree == null || m_dirTree.getRowCount()==0)
            return null;
        return this.selectDirForPathInternal(treePath.getPath(), bExact);
    } // selectDirForPath

    /**
     * Private method that tries to select the directory node denoted by the
     * object array passed as parameter.
     * @param objPath - Tree path object array of the directory to be selected.
     * Each element is a string returned from a
     * {@link DirectoryObject#toString()} call.
     * @param bExact - <tt>True</tt> if an exact match is needed,
     * <tt>false</tt> if an ancestor of the node denoted by the tree path is
     * sufficient
     * @return The actually selected directory object, or <tt>null</tt>
     * otherwise
     */
    private DirectoryObject selectDirForPathInternal
        ( Object[] objPath
        , boolean  bExact
        )
    {
        DirectoryObject[]         seriesDirs;
        DirectoryObject           dirObj, d;
        List<Object>              newPathList;
        Iterator<DirectoryObject> subDirIter;
        String                    str;
        int                       i;

        if (objPath == null || objPath.length < 2)
            return null;
        dirObj = null;
        newPathList = new ArrayList<Object>(objPath.length);
        newPathList.add(m_dirTreeModel.getRoot());
        // Get the series directory.
        m_dirTree.expandRow(0);
        seriesDirs = this.getFrame().getSeriesContainer().
            getSeriesDirectories();
        str = objPath[1].toString();
        for (i = 0; i < seriesDirs.length; i++)
        {
            if (seriesDirs[i].getFileName().equalsIgnoreCase(str))
            {
                dirObj = seriesDirs[i];
                newPathList.add(seriesDirs[i]);
                break;
            }
        }
        if (dirObj == null)
            return null;
        // Get the remaining path components.
        for (i = 2; dirObj != null && i < objPath.length; i++)
        {
            m_dirTree.expandPath(new TreePath(newPathList.toArray()));
            str = objPath[i].toString();
            subDirIter = dirObj.getSubDirIterator();
            if (bExact)
                dirObj = null;
            while (subDirIter.hasNext())
            {
                d = subDirIter.next();
                if (d.getFileName().equalsIgnoreCase(str))
                {
                    dirObj = d;
                    newPathList.add(d);
                    break;
                }
            }
        }
        // Select the node.
        if (dirObj != null)
            m_dirTree.setSelectionPath(new TreePath(newPathList.toArray()));
        return dirObj;
    } // selectDirForPath

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#saveChanges()
     */
    public void saveChanges ()
    {
    } // saveChanges

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#refresh()
     */
    public void refresh ()
    {
        m_dirTreeModel.refresh(this.getCurrentDirectory());
        // Get the currently selected tree node.
        TreePath treePath = m_dirTree.getSelectionPath();
        m_dirTreeModel.refresh(null);
        // Try to select the previously selected node.
        DirectoryObject selDir = this.selectDirForPath(treePath, false);
        if (selDir != null)
        {
            // Set this to the current directory.
            if (selDir != this.getCurrentDirectory())
            {
                this.setCurrentDirectory(selDir);
                this.fireCurrentDirectoryChanged(selDir);
            }
        }
        else
        {
            // If nothing has been selected, go to the directory tree root.
            if (m_dirTree.getRowCount() > 0)
            {
                // Select the root node.
                m_dirTree.expandRow(0);
                m_dirTree.setSelectionRow(0);
            }
        }
    } // refresh

    /**
     * Refreshes one directory object.
     * @param dirObject - The directory
     */
    public void refresh (DirectoryObject dirObject)
    {
        m_dirTreeModel.refresh(dirObject);
    } // refresh

    /**
     * Called to inform us that the directory structure has been changed.
     * @param dirObject - The directory object
     */
    public void onTreeStructureChanged (DirectoryObject dirObject)
    {
        m_dirTreeModel.fireTreeStructureChanged(dirObject);
    } // onTreeStructureChanged

    /**
     * Private method to create the directory context menu. It is a separate
     * method to facilitate advising using AspectJ.
     */
    private void createDirContextMenu ()
    {
        if (m_dirPopupMenu != null)
            return;
        // Create the "Directory" context menu.
        m_dirPopupMenu = new JPopupMenu("Directory Menu");
        m_dpNewDir = new JMenuItem("New Subfolder ...");
        m_dpNewDir.addActionListener(this);
        m_dpProperties = new JMenuItem("Properties");
        m_dpProperties.addActionListener(this);
        m_dpExport = new JMenuItem("Export");
        m_dpExport.addActionListener(this);
        m_dpCut = new JMenuItem("Cut");
        m_dpCut.addActionListener(this);
        m_dpPaste = new JMenuItem("Paste");
        m_dpPaste.addActionListener(this);
        m_dpDelete = new JMenuItem("Delete");
        m_dpDelete.addActionListener(this);
        // Add the menu items.
        m_dirPopupMenu.add(m_dpProperties);
        JSeparator sep = new JSeparator();
        sep.setBackground(new Color(255, 255, 255));
        sep.setForeground(new Color(153, 153, 153));
        m_dirPopupMenu.add(sep);
        m_dirPopupMenu.add(m_dpExport);
        sep = new JSeparator();
        sep.setBackground(new Color(255, 255, 255));
        sep.setForeground(new Color(153, 153, 153));
        m_dirPopupMenu.add(sep);
        m_dirPopupMenu.add(m_dpNewDir);
        sep = new JSeparator();
        sep.setBackground(new Color(255, 255, 255));
        sep.setForeground(new Color(153, 153, 153));
        m_dirPopupMenu.add(sep);
        m_dirPopupMenu.add(m_dpCut);
        m_dirPopupMenu.add(m_dpPaste);
        sep = new JSeparator();
        sep.setBackground(new Color(255, 255, 255));
        sep.setForeground(new Color(153, 153, 153));
        m_dirPopupMenu.add(sep);
        m_dirPopupMenu.add(m_dpDelete);
    } // createDirContextMenu

    /**
     * This method is invoked by the directory tree mouse listener to show the
     * directory context menu.
     * @param x - Mouse x coordinate
     * @param y - Mouse y coordinate
     */
    public void showDirContextMenu (int x, int y)
    {
        int rowIndex = m_dirTree.getRowForLocation(x, y);
        if (rowIndex >= 0)
            m_dirTree.setSelectionRow(rowIndex);
        if (this.getCurrentDirectory() == null)
            return;
        this.createDirContextMenu();
        m_dpPaste.setEnabled(this.getFrame().canPaste());
        // Show the context menu.
        m_dirPopupMenu.show(m_dirTree, x, y);
    } // showDirContextMenu

    /**
     * This method is invoked for the <b>Directory</b> context menu actions.
     * @param e - The event specifying which context menu item has been selected
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    public void actionPerformed (ActionEvent e)
    {
        Object          evtSource  = e.getSource();
        DirectoryObject currentDir = this.getCurrentDirectory();

        if (evtSource == m_dpProperties)
        {
            if (this.getFrame().showProperties(currentDir))
            {
                m_dirTreeModel.fireTreeStructureChanged(currentDir);
                this.firePropertiesChanged(currentDir);
            }
        }
        else if (evtSource == m_dpExport)
        {
            if (currentDir == null)
                return;
            this.getFrame().startExport(currentDir, true, true, false);
        }
        else if (evtSource == m_dpNewDir)
        {
            this.getFrame().newDirectory(currentDir);
        }
        else if (evtSource == m_dpCut)
        {
            if (currentDir == null)
                return;
            // Cut the current directory.
            List<AbstractFSObject> fsObjs = new ArrayList<AbstractFSObject>(1);
            fsObjs.add(this.getCurrentDirectory());
            this.getFrame().cut(fsObjs);
        }
        else if (evtSource == m_dpPaste)
        {
            this.getFrame().paste();
        }
        else if (evtSource == m_dpDelete)
        {
            if (currentDir == null)
                return;
            // Delete the current directory.
            int res = JOptionPane.showConfirmDialog(this,
                "Do you really want to delete the directory \""
                + currentDir.getFileName()
                + "\" \nand destroy its complete contents?",
                AppInfo.APP_NAME, JOptionPane.YES_NO_OPTION,
                JOptionPane.QUESTION_MESSAGE);
            if (res != 0)
                return;
            if (!currentDir.delete(true))
            {
                // Restore what is left.
                try
                {
                    this.getFrame().getSeriesContainer().loadSeries(currentDir);
                    // Show the warning dialog.
                    JOptionPane.showMessageDialog(this,
                       "Cannot delete directory \""+ currentDir.getFileName()
                       + "\"", AppInfo.APP_NAME, JOptionPane.WARNING_MESSAGE);
                }
                catch (PersistException pe)
                {
                    // Show the load error dialog.
                    JOptionPane.showMessageDialog(this, pe.getMessage(),
                        AppInfo.APP_NAME, JOptionPane.ERROR_MESSAGE);
                }
            }
            this.refresh();
            this.firePropertiesChanged(currentDir);
        }
    } // actionPerformed

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#loadSettings(com.dgrossmann.photo.settings.Settings)
     */
    public void loadSettings (Settings settings)
    {
        String          currentDirStr;
        StringTokenizer tok;
        Object[]        currentDirPath;
        int             i;

        m_bSaveCurrentDirOnExit = settings.getBoolean(SAVE_CURRENT_DIR_ON_EXIT,
            false);
        if (!m_bSaveCurrentDirOnExit)
            return;
        // Select the current directory.
        currentDirStr = settings.get(CURRENT_DIR);
        if (currentDirStr == null || currentDirStr.length() == 0)
            return;
        tok = new StringTokenizer(currentDirStr, "\n");
        currentDirPath = new Object[tok.countTokens()];
        i = 0;
        while (tok.hasMoreTokens())
            currentDirPath[i++] = tok.nextToken();
        this.selectDirForPathInternal(currentDirPath, false);
    } // loadSettings

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#saveSettings(com.dgrossmann.photo.settings.Settings)
     */
    public void saveSettings (Settings settings)
    {
        Object[] currentDirPath;
        int      i;
        String   str;

        settings.setBoolean(SAVE_CURRENT_DIR_ON_EXIT, m_bSaveCurrentDirOnExit);
        if (!m_bSaveCurrentDirOnExit || this.getCurrentDirectory() == null)
        {
            settings.remove(CURRENT_DIR);
            return;
        }
        currentDirPath = m_dirTree.getSelectionPath().getPath();
        str = "";
        for (i = 0; i < currentDirPath.length; i++)
        {
            str += currentDirPath[i].toString();
            if (i < currentDirPath.length - 1)
                str += "\n";
        }
        settings.set(CURRENT_DIR, str);
    } // saveSettings

    /**
     * @see java.awt.event.FocusListener#focusGained(java.awt.event.FocusEvent)
     */
    public void focusGained (FocusEvent e)
    {
        if (this.getCurrentDirectory() != null)
        {
            List<AbstractFSObject> ls = new ArrayList<AbstractFSObject>(1);
            ls.add(this.getCurrentDirectory());
            this.fireSelectionChanged(ls);
        }
    } // focusGained

    /**
     * @see java.awt.event.FocusListener#focusLost(java.awt.event.FocusEvent)
     */
    public void focusLost (FocusEvent e)
    {
    } // focusLost
} // DirTreePanel
