/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: PanelTitleComponent
    Created: 13 June, 2003
        $Id: PanelTitleComponent.java 57 2006-02-14 21:01:38Z Dirk $
  $Revision: 57 $
      $Date: 2006-02-14 22:01:38 +0100 (Di, 14 Feb 2006) $
    $Author: Dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui.panel;

import java.awt.*;

import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JPanel;

import com.dgrossmann.photo.ui.ImageHolder;

/**
 * Provides a label with a gradient background.
 * @author Dirk Grossmann
 */
public class PanelTitleComponent extends JPanel
{
    private static final String PREVIEW_TOOL_TIP     = "Show the preview panel";
    private static final String PREVIEW_SEL_TOOL_TIP = "Hide the preview panel";

    public static final Color COL_FOREGROUND         = Color.WHITE;
    public static final Color COL_ACTIVE_BLUE_LIGHT  = new Color(133, 166, 255);
    public static final Color COL_ACTIVE_BLUE_DARK   = new Color(44, 66, 222);
    public static final Color COL_ACTIVE_RED_LIGHT   = new Color(222, 155, 177);
    public static final Color COL_ACTIVE_RED_DARK    = new Color(211, 44, 88);
    public static final Color COL_INACTIVE_RED_LIGHT = new Color(202, 188, 177);
    public static final Color COL_INACTIVE_BLUE_LIGHT= new Color(185, 185,192);
    public static final Color COL_INACTIVE_DARK      = new Color(166, 155, 144);
    public static final Color COL_BOTTOM_LINE        = new Color(133, 133, 133);

    public static final int ACTIVE_BG_LEFT_COLOR       = 1;
    public static final int ACTIVE_BG_MIDDLE_COLOR     = 2;
    public static final int INACTIVE_BG_LEFT_COLOR     = 3;
    public static final int INACTIVE_BG_MIDDLE_COLOR   = 4;
    public static final int ACTIVE_BOTTOM_LINE_COLOR   = 5;
    public static final int INACTIVE_BOTTOM_LINE_COLOR = 6;

    public static final String CARD_NORMAL             = "normal";
    public static final String CARD_PREVIEW            = "preview";

    private static ImageIcon s_previewIcon    = null;
    private static ImageIcon s_previewSelIcon = null;

    private boolean                  m_bIsActive;
    private Icon                     m_titleImage;
    private String                   m_title;
    private boolean                  m_bIsPreviewSelected;
    private JPanel                   m_parentPanel,m_normalPanel,m_previewPanel;
    private PanelTitleChangeListener m_changeListener;

    private Color                    m_fgColor;
    private Color                    m_stdFgColor;
    private Color                    m_topBgColor;
    private Color                    m_middleBgColor;
    private Color                    m_stdActiveTopBgColor;
    private Color                    m_stdInactiveTopBgColor;
    private Color                    m_stdActiveMiddleBgColor;
    private Color                    m_stdInactiveMiddleBgColor;
    private Color                    m_bottomLineColor;
    private Color                    m_stdActiveBottomLineColor;
    private Color                    m_stdInactiveBottomLineColor;
    private boolean                  m_bSimpleGradient;
    private boolean                  m_bHasPainted;

    /** Constructor of a PanelTitleComponent instance. */
    public PanelTitleComponent ()
    {
        this(" ", null);
    } // PanelTitleComponent
    
    /**
     * Constructor of a PanelTitleComponent instance.
     * @param title - Title text to be displayed
     */
    public PanelTitleComponent (String title)
    {
        this(title, null);
    } // PanelTitleComponent
    
    /**
     * Constructor of a PanelTitleComponent instance.
     * @param title - Title text to be displayed
     * @param image - Icon image
     */
    public PanelTitleComponent (String title, Icon image)
    {
        super();
        m_bHasPainted = false;
        m_bIsPreviewSelected = false;
        m_titleImage = image;
        m_title = title;
        if (m_title == null || m_title.length() == 0)
            m_title = " ";
        m_parentPanel = m_normalPanel = m_previewPanel = null;
        m_changeListener = null;
        // Component initialization.
        this.initComponents();
        // Set the preview icon.
        if (s_previewIcon == null)
        {
            s_previewIcon = ImageHolder.loadStandardImage(
                "images/preview.gif", this);
            s_previewSelIcon = ImageHolder.loadStandardImage(
                "images/preview_sel.gif", this);
        }
        // Set the title string.
        titleLabel.setText(m_title);
        // Set the color scheme.
        boolean bRedColors = false;
        if (!"Windows XP".equalsIgnoreCase(System.getProperty("os.name")))
        {
            bRedColors = SystemColor.activeCaption.getRed() >
                SystemColor.activeCaption.getBlue();
        }
        // Initialize the standard colors.
        m_stdFgColor = COL_FOREGROUND;
        if (bRedColors)
            this.setRedStandardColors();
        else
            this.setBlueStandardColors();
        m_stdInactiveTopBgColor = COL_INACTIVE_DARK;
        m_stdActiveBottomLineColor = COL_BOTTOM_LINE;
        m_stdInactiveBottomLineColor = COL_BOTTOM_LINE;
        // Initialize as active sinple gradient.
        m_bIsActive = true;
        this.setActiveColors();
        this.setCaneGradient(false);
        // Set the standard label properties.
        Font fn = titleLabel.getFont();
        Font newFont = fn.deriveFont(Font.BOLD);
        titleLabel.setFont(newFont);
        titleLabel.setForeground(m_fgColor);
        titleLabel.setOpaque(false);
    } // PanelTitleComponent

    /**
     * Sets the panel components.
     * @param parentPanel - The parent panel that will either show the
     * <tt>normalPanel</tt> or the <tt>previewPanel</tt>
     * @param normalPanel - The normal panel
     * @param previewPanel - The preview panel
     * @param bSelectPreviewPanel - <tt>True</tt> to select the preview panel
     */
    public void setPanelComponents
        ( JPanel  parentPanel
        , JPanel  normalPanel
        , JPanel  previewPanel
        , boolean bSelectPreviewPanel
        )
    {
        m_parentPanel = parentPanel;
        m_normalPanel = normalPanel;
        m_previewPanel = previewPanel;
        this.setPreview(bSelectPreviewPanel);
    } // setPanelComponents

    /**
     * Sets a new title text.
     * @param title - New title string
     */
    public void setText (String title)
    {
        if (title == null || title.length() == 0)
            m_title = " ";
        else
            m_title = title;
        titleLabel.setText(m_title);
    } // setText

    /**
     * Gets whether the preview panel is selected.
     * @return <tt>True</tt> iff the preview panel is selected
     */
    public boolean isPreview ()
    {
        return m_bIsPreviewSelected;
    } // isPreview

    /**
     * Sets whether the preview panel is selected.
     * @param bIsPreview - <tt>True</tt> to select the preview panel
     */
    public void setPreview (boolean bIsPreview)
    {
        // Do nothing if we do not have panels.
        if (m_parentPanel == null)
            return;
        // Switch the image as appropriate.
        m_bIsPreviewSelected = bIsPreview;
        previewCtrl.setIcon(m_bIsPreviewSelected ?
            s_previewSelIcon : s_previewIcon);
        previewCtrl.setToolTipText(m_bIsPreviewSelected ?
            PREVIEW_SEL_TOOL_TIP : PREVIEW_TOOL_TIP);
        // Switch the panel.
        if (m_bIsPreviewSelected)
        {
            ((CardLayout) m_parentPanel.getLayout()).show(
                m_previewPanel.getParent(), CARD_PREVIEW);
        }
        else
        {
            ((CardLayout) m_parentPanel.getLayout()).show(
                m_previewPanel.getParent(), CARD_NORMAL);
        }
    } // setPreview

    /**
     * Adds a change listener.
     * @param changeListener - The listener being informed when the selection
     * changes
     */
    public void addChangeListener (PanelTitleChangeListener changeListener)
    {
        m_changeListener = changeListener;
    } // addChangeListener

    /**
     * Sets the active colors.
     */
    public void setActiveColors ()
    {
        m_bIsActive = true;
        m_fgColor = m_stdFgColor;
        m_topBgColor = m_stdActiveTopBgColor;
        m_middleBgColor = m_stdActiveMiddleBgColor;
        m_bottomLineColor = m_stdActiveBottomLineColor;
        if (this.isShowing())
            this.repaint();
    } // setActiveColors

    /**
     * Sets the inactive colors.
     */
    public void setInactiveColors ()
    {
        m_bIsActive = false;
        m_fgColor = m_stdFgColor;
        m_topBgColor = m_stdInactiveTopBgColor;
        m_middleBgColor = m_stdInactiveMiddleBgColor;
        m_bottomLineColor = m_stdInactiveBottomLineColor;
        if (this.isShowing())
            this.repaint();
    } // setInactiveColors

    /**
     * Gets a nice brighter and less saturated color.
     * @param color - Start color
     * @return The brighter color
     */
    public Color getBrighterColor (Color color)
    {
        Color col;
        int   avgVal = (color.getRed() + color.getGreen() + color.getBlue())/3;

        col = color.brighter();
        if (avgVal < 200)
            col = col.brighter();
        return col;
    } // getBrighterColor

    /**
     * Sets the gradient style: cane or simple.
     * @param bCane - <tt>True</tt> for cane gradient, <tt>false</tt> for
     * simple gradient
     */
    public void setCaneGradient (boolean bCane)
    {
        m_bSimpleGradient = !bCane;
        if (m_bHasPainted)
        {
            Dimension siz = this.getSize();
            if (siz.width > 0 && siz.height > 0)
                this.repaint(new Rectangle(siz));
        }
    } // setCaneGradient

    /**
     * Sets the background color.
     * @param whichColor - <tt>BG_LEFT_COLOR</tt> to set the left color,
     * <tt>BG_RIGHT_COLOR</tt> to set the right color of the gradient.
     * <tt>BOTTOM_LINE_COLOR</tt> to set the color of the line drawn at the
     * bottom (or <tt>null</tt> "color" for no line).
     * @param newColor - Color to be used (can be <tt>null</tt> in combination
     * with <tt>BOTTOM_LINE_COLOR</tt>)
     */
    public void setBackground (int whichColor, Color newColor)
    {
        if (newColor == null)
            return;
        switch (whichColor)
        {
        case ACTIVE_BG_LEFT_COLOR:
            m_stdActiveTopBgColor = newColor;
            break;
        case ACTIVE_BG_MIDDLE_COLOR:
            if (newColor != null)
            m_stdActiveMiddleBgColor = newColor;
            break;
        case ACTIVE_BOTTOM_LINE_COLOR:
            m_stdActiveBottomLineColor = newColor;
            break;
        case INACTIVE_BG_LEFT_COLOR:
            m_stdInactiveTopBgColor = newColor;
            break;
        case INACTIVE_BG_MIDDLE_COLOR:
            if (newColor != null)
            m_stdInactiveMiddleBgColor = newColor;
            break;
        case INACTIVE_BOTTOM_LINE_COLOR:
            m_stdInactiveBottomLineColor = newColor;
            break;
        }
        if (m_bHasPainted)
        {
            Dimension siz = this.getSize();
            if (siz.width > 0 && siz.height > 0)
                this.repaint(new Rectangle(siz));
        }
    } // setBackground

    /** Sets the red standard colors. */
    public void setRedStandardColors ()
    {
        m_stdActiveTopBgColor = COL_ACTIVE_RED_DARK;
        m_stdActiveMiddleBgColor = COL_ACTIVE_RED_LIGHT;
        m_stdInactiveMiddleBgColor = COL_INACTIVE_RED_LIGHT;
        if (m_bHasPainted)
        {
            Dimension siz = this.getSize();
            if (siz.width > 0 && siz.height > 0)
                this.repaint(new Rectangle(siz));
        }
    } // setRedStandardColors

    /** Sets the blue standard colors. */
    public void setBlueStandardColors ()
    {
        m_stdActiveTopBgColor = COL_ACTIVE_BLUE_DARK;
        m_stdActiveMiddleBgColor = COL_ACTIVE_BLUE_LIGHT;
        m_stdInactiveMiddleBgColor = COL_INACTIVE_BLUE_LIGHT;
        if (m_bHasPainted)
        {
            Dimension siz = this.getSize();
            if (siz.width > 0 && siz.height > 0)
                this.repaint(new Rectangle(siz));
        }
    } // setBlueStandardColors

    /**
     * Paints the component.
     * @param g - Graphics context
     */
    protected void paintComponent (Graphics g)
    {
        Graphics2D    g2d = (Graphics2D) g;
        Dimension     siz = this.getSize();
        float         middleY, bottomY;
        GradientPaint paint;
        
        // Paint the background gradient.
        if (siz.width > 0 && siz.height > 0)
        {
            middleY = siz.height / 2 - 2;
            bottomY = siz.height - 3;
            Paint oldPaint = g2d.getPaint();
            if (m_bSimpleGradient)
            {
                paint = new GradientPaint(0.0F, 0.0F, m_middleBgColor,
                    0.0F, bottomY, m_topBgColor);
                g2d.setPaint(paint);
                g2d.fillRect(0, 0, siz.width, siz.height-1);
            }
            else
            {
                int middleYint = (int) middleY;
                paint = new GradientPaint(0.0F, 0.0F, m_topBgColor,
                    0.0F, middleY, m_middleBgColor);
                g2d.setPaint(paint);
                g2d.fillRect(0, 0, siz.width, middleYint);
                paint = new GradientPaint(0.0F, middleY, m_middleBgColor,
                    0.0F, bottomY, m_topBgColor);
                g2d.setPaint(paint);
                g2d.fillRect(0, middleYint, siz.width, siz.height-2);
            }
            g2d.setPaint(oldPaint);
            if (m_bottomLineColor != null)
            {
                Color oldColor = g2d.getColor();
                g2d.setColor(m_bottomLineColor);
                g2d.drawLine(0, siz.height-1, siz.width-1, siz.height-1);
                g2d.setColor(oldColor);
            }
            m_bHasPainted = true;
        }
        // Do not call the base class' method.
    } // paintComponent

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents()
    {
        java.awt.GridBagConstraints gridBagConstraints;

        titleLabel = new javax.swing.JLabel();
        previewCtrl = new javax.swing.JLabel();

        setLayout(new java.awt.GridBagLayout());

        titleLabel.setText("Title");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        add(titleLabel, gridBagConstraints);

        previewCtrl.setMaximumSize(new java.awt.Dimension(60, 16));
        previewCtrl.setMinimumSize(new java.awt.Dimension(60, 16));
        previewCtrl.setPreferredSize(new java.awt.Dimension(60, 16));
        previewCtrl.addMouseListener(new java.awt.event.MouseAdapter()
        {
            public void mouseClicked(java.awt.event.MouseEvent evt)
            {
                onPreviewCtrlClicked(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        add(previewCtrl, gridBagConstraints);

    }// </editor-fold>//GEN-END:initComponents

    private void onPreviewCtrlClicked (java.awt.event.MouseEvent evt)//GEN-FIRST:event_onPreviewCtrlClicked
    {//GEN-HEADEREND:event_onPreviewCtrlClicked
        // Do nothing if not active or not showing two panels.
        if (m_parentPanel == null)
            return;
        // Switch the image as appropriate.
        this.setPreview(!m_bIsPreviewSelected);
        if (m_changeListener != null)
        {
            m_changeListener.onTitleSelectionChanged(m_bIsPreviewSelected ?
                PanelTitleChangeListener.SEL_PREVIEW :
                PanelTitleChangeListener.SEL_NORMAL);
        }
    }//GEN-LAST:event_onPreviewCtrlClicked

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JLabel previewCtrl;
    private javax.swing.JLabel titleLabel;
    // End of variables declaration//GEN-END:variables
} // PanelTitleComponent
