/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: ContentsPanel
    Created: 17 December, 2005
        $Id: ContentsPanel.java 160 2009-05-31 07:57:29Z dirk $
  $Revision: 160 $
      $Date: 2009-05-31 09:57:29 +0200 (So, 31 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui.panel.contents;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.JScrollBar;
import javax.swing.JSeparator;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import com.dgrossmann.photo.dir.AbstractFSObject;
import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.settings.Settings;
import com.dgrossmann.photo.ui.ExplorerMainFrame;
import com.dgrossmann.photo.ui.ShellExec;
import com.dgrossmann.photo.ui.dialog.AssociateDialog;
import com.dgrossmann.photo.ui.dialog.ExportedImageViewDialog;
import com.dgrossmann.photo.ui.dialog.PropertiesDialog;
import com.dgrossmann.photo.ui.panel.AbstractExplorerPanel;
import com.dgrossmann.photo.ui.panel.PanelTitleChangeListener;
import com.dgrossmann.photo.webexport.ExportException;
import com.dgrossmann.photo.webexport.ExportFactory;
import com.dgrossmann.photo.webexport.IWebExport;

/**
 * Panel class for the contents panel.
 * @author Dirk Grossmann
 */
public class ContentsPanel extends AbstractExplorerPanel
    implements ListSelectionListener, ActionListener, FocusListener,
    PanelTitleChangeListener
{
    // Setting names being used in the settings dialog.
    public static final String DBL_CLICK_DIR_OPENS  =
        "contents.dbl_click_on_dir_opens";
    public static final String DBL_CLICK_FILE_OPENS =
        "contents.dbl_click_on_file_opens";
    private static final String CONT_COLUMN_POS     = "contents.column_pos.";
    private static final String CONT_SHOW_PREVIEW   = "contents.show_preview";

    // Title texts.
    private static final String CONTENTS_PANE_TITLE = "Contents";
    private static final String NEWITEM_FILE        = "File";
    private static final String NEWITEM_SUBGROUP    = "Subgroup";
    private static final String NEWITEM_REFERENCE   = "Reference";
    private static final String NEWITEM_SEPARATOR   = "Separator";

    // Default table part sizes.
    private static final int SCREEN_WIDTH_THRESH    = 1200;
    private static final int TABLE_ROW_SIZE         =   52;
    private static final int SZ_IMAGE_WIDTH_L       = TABLE_ROW_SIZE + 10;
    private static final int SZ_NAME_S              =  180;
    private static final int SZ_NAME_L              =  250;
    private static final int SZ_MODIFIED_WIDTH_S    =  102;
    private static final int SZ_MODIFIED_WIDTH_L    =  144;
    private static final int SZ_EXPORT_WIDTH_S      =   48;
    private static final int SZ_EXPORT_WIDTH_L      =   58;

    private ContentsTableModel     m_fileTableModel;
    private boolean                m_bDblClickOnDirOpens;
    private boolean                m_bDblClickOnFileOpens;
    private boolean                m_bIsPreviewShowing;
    private ContentsPreviewManager m_previewManager;

    // File popup menu and items.
    private JPopupMenu             m_filePopupMenu;
    private JMenuItem              m_fpOpen, m_fpShowExported, m_fpProperties;
    private JMenuItem              m_fpCut, m_fpAssociate, m_fpDelete;
    private JMenuItem              m_fpMoveUp, m_fpMoveDown;

    /**
     * Creates a new <tt>ContentsPanel</tt> instance.
     * @param frm - The parent explorer frame
     */
    public ContentsPanel (ExplorerMainFrame frm)
    {
        // Set up the member variables.
        super(frm, CONTENTS_PANE_TITLE);
        m_fileTableModel = new ContentsTableModel
            (this.getRootComponent(), this, frm.getImageHolder());
        m_filePopupMenu = null;
        m_bIsPreviewShowing = false;
        // Initialize the components.
        this.initComponents();
        this.add(this.getTitleLabel(), BorderLayout.NORTH);
        m_previewManager = new ContentsPreviewManager(this, htmlPane);
        // Fill the "New Object" combo box.
        newObjSelComboBox.addItem(NEWITEM_FILE);
        newObjSelComboBox.addItem(NEWITEM_SUBGROUP);
        newObjSelComboBox.addItem(NEWITEM_REFERENCE);
        newObjSelComboBox.addItem(NEWITEM_SEPARATOR);
        newObjSelComboBox.setSelectedItem(NEWITEM_SEPARATOR);
        // Add the listeners.
        ListSelectionModel lsModel = listTable.getSelectionModel();
        lsModel.setSelectionMode
            (ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
        lsModel.addListSelectionListener(this);
        listTable.addMouseListener(new ContentsTableMouseListener(this));
        listTable.addFocusListener(this);
        // At last, set the null current directory.
        this.setCurrentDirectory(null);
    } // ContentsPanel

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#setupComponents()
     */
    public void setupComponents ()
    {
        Toolkit          tk         = Toolkit.getDefaultToolkit();
        Dimension        screenSize = tk.getScreenSize();
        TableColumnModel cModel     = listTable.getColumnModel();
        JTableHeader     thead      = listTable.getTableHeader();
        Dimension        thSize     = thead.getPreferredSize();
        TableColumn      col;
        int              colWidth, sumColWidths;

        // Complete the table component by setting the default sizes.
        if (thSize.height < 18)
            thSize.height = 18;
        thead.setPreferredSize(thSize);
        listTable.setRowHeight(TABLE_ROW_SIZE);
        listTable.setRowMargin(1);
        listTable.setShowHorizontalLines(true);
        listTable.setGridColor(new Color(233, 233, 222));
        sumColWidths = 0;
        // The "Preview/Type" column (fixed according to the image size).
        colWidth = (screenSize.width > SCREEN_WIDTH_THRESH) ?
        	SZ_IMAGE_WIDTH_L : TABLE_ROW_SIZE;
        col = cModel.getColumn(0);
        col.setMinWidth(colWidth);
        col.setPreferredWidth(colWidth);
        sumColWidths += colWidth;
        // The "Name" column.
        colWidth = (screenSize.width > SCREEN_WIDTH_THRESH) ?
        	SZ_NAME_L : SZ_NAME_S;
        col = cModel.getColumn(1);
        col.setMinWidth(0);
        col.setPreferredWidth(colWidth);
        sumColWidths += colWidth;
        // The "Modified" column.
        colWidth = (screenSize.width > SCREEN_WIDTH_THRESH) ?
            SZ_MODIFIED_WIDTH_L : SZ_MODIFIED_WIDTH_S;
        col = cModel.getColumn(3);
        col.setMinWidth(0);
        col.setPreferredWidth(colWidth);
        sumColWidths += colWidth;
        // The "Export" column.
        colWidth = (screenSize.width > SCREEN_WIDTH_THRESH) ?
            SZ_EXPORT_WIDTH_L : SZ_EXPORT_WIDTH_S;
        col = cModel.getColumn(4);
        col.setMinWidth(0);
        col.setPreferredWidth(colWidth);
        sumColWidths += colWidth;
        // Subtract the size of a scrollbar.
        JScrollBar tempSB = new JScrollBar();
        Dimension prefSBSize = tempSB.getUI().getPreferredSize(tempSB);
        if (prefSBSize != null)
            sumColWidths += prefSBSize.width;
        // The "Title" column gets the rest.
        if (thead.getWidth() > sumColWidths)
        {
            col = cModel.getColumn(2);
            col.setMinWidth(0);
            col.setPreferredWidth(thead.getWidth() - sumColWidths);
        }
    } // setupComponents

    /**
     * @see com.dgrossmann.photo.ui.panel.AbstractExplorerPanel#setActiveColors()
     */
    @Override
	public void setActiveColors ()
    {
        if (this.getCurrentDirectory() != null)
            super.setActiveColors();
        else
            super.setInactiveColors();
    } // setActiveColors

    /**
     * @see com.dgrossmann.photo.ui.panel.AbstractExplorerPanel#setCurrentDirectory(com.dgrossmann.photo.dir.DirectoryObject)
     */
    @Override
	public void setCurrentDirectory (DirectoryObject currentDirectory)
    {
        super.setCurrentDirectory(currentDirectory);
        // Enable the buttons.
        newObjSelComboBox.setEnabled(currentDirectory != null);
        newObjButton.setEnabled(currentDirectory != null);
        propertiesButton.setEnabled(false);
        pasteButton.setEnabled(this.getFrame().canPaste());
        delButton.setEnabled(false);
        orderByFileNameButton.setEnabled(currentDirectory != null &&
        	currentDirectory.getFileCount() > 1);
        moveUpButton.setEnabled(false);
        moveDownButton.setEnabled(false);
        // Refresh the table.
        this.refresh();
        m_previewManager.setCurrentDirectory(currentDirectory);
    } // setCurrentDirectory

    /**
     * Adds a file system object to the table and select it.
     * @param fsObj - The file system object to add
     */
    public void addNewFSObject (AbstractFSObject fsObj)
    {
        m_fileTableModel.addFSObject(fsObj, (fsObj instanceof DirectoryObject) ?
            -1 : listTable.getSelectedRow());
        this.setCurrentFSObject(fsObj);
        this.selectCurrentFiles(false, null);
    } // addNewFSObject

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#saveChanges()
     */
    public void saveChanges ()
    {
    } // saveChanges

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#refresh()
     */
    public void refresh ()
    {
        DirectoryObject currentDirectory = this.getCurrentDirectory();
        String          str, dirName, title;

        m_fileTableModel.setCurrentDir(currentDirectory);
        if (currentDirectory == null)
        {
            this.setInactiveColors();
            this.setTitle(CONTENTS_PANE_TITLE);
            pasteButton.setEnabled(false);
            return;
        }
        // Set the title.
        this.setActiveColors();
        pasteButton.setEnabled(this.getFrame().canPaste());
        if (currentDirectory.getParent() == null)
            str = "Contents of series \"";
        else
            str = "Contents of group \"";
        title = currentDirectory.getTitlePlain();
        dirName = currentDirectory.getFileName();
        str += title + "\"";
        if (title != null && !title.equals(dirName))
            str += " - Directory \"" + dirName + "\"";
        this.setTitle(str);
    } // refresh

    /**
     * @see com.dgrossmann.photo.ui.panel.PanelTitleChangeListener#onTitleSelectionChanged(int)
     */
    public void onTitleSelectionChanged (int panelId)
    {
        m_bIsPreviewShowing = (panelId == PanelTitleChangeListener.SEL_PREVIEW);
        m_previewManager.setActive(m_bIsPreviewShowing);
    } // onTitleSelectionChanged

    /**
     * Selects the rows for the current file objects or the objects passed as
     * parameter.
     * @param bRefresh - <tt>True</tt> to refresh the table before selecting
     * @param fileObjs - Collection of the file system objects to select or
     * <tt>null</tt> to use the current file system objects.
     */
    public void selectCurrentFiles
    	( boolean                bRefresh
    	, List<AbstractFSObject> fileObjs
    	)
    {
        AbstractFSObject           fsObj;
        Iterator<AbstractFSObject> iter;
        int                        index, firstIndex;

        if (listTable.getRowCount() == 0)
            return;
        if (fileObjs == null)
        {
            // Save the current file object data.
            fileObjs = new ArrayList<AbstractFSObject>(40);
            if (this.getCurrentFSObjects() != null)
            	fileObjs.addAll(this.getCurrentFSObjects());
        }
        if (bRefresh)
            m_fileTableModel.refresh();
        firstIndex = -1;
        listTable.clearSelection();
        if (fileObjs.size() >= 1)
        {
            iter = fileObjs.iterator();
            while (iter.hasNext())
            {
                fsObj = (iter.next());
                index = m_fileTableModel.getIndexOfFSObject(fsObj);
                if (index >= 0)
                {
                    listTable.addRowSelectionInterval(index, index);
                    if (firstIndex < 0)
                        firstIndex = index;
                }
            }
        }
        else
        {
            firstIndex = 0;
            // Select the first row in the table - this will also adjust
            // "m_currentFileObjs".
            listTable.setRowSelectionInterval(firstIndex, firstIndex);
        }
        // Make the first selected row visible.
        if (firstIndex >= 0)
        {
            // ... (?)
        }
    } // selectCurrentFiles

    /**
     * @see javax.swing.event.ListSelectionListener#valueChanged(javax.swing.event.ListSelectionEvent)
     */
    public void valueChanged (ListSelectionEvent e)
    {
        DirectoryObject currentDir = this.getCurrentDirectory();
        newObjSelComboBox.setEnabled(currentDir != null);
        newObjButton.setEnabled(currentDir != null);
        delButton.setEnabled(false);
        this.setCurrentFSObjects(null);
        if (currentDir == null)
            return;
        int[] selRows = listTable.getSelectedRows();
        if (selRows.length == 0)
            return;
        List<AbstractFSObject> currentFSObjs = new ArrayList<AbstractFSObject>();
        for (int i = 0; i < selRows.length; i++)
            currentFSObjs.add(m_fileTableModel.getFSObjectAt(selRows[i]));
        this.setCurrentFSObjects(currentFSObjs);
        // Enable the rest of the file buttons.
        delButton.setEnabled(true);
        if (currentFSObjs.size() == 1)
        {
            boolean bIsSeparator = false;
            if (currentFSObjs.get(0) instanceof FileObject &&
                ((FileObject) (currentFSObjs.get(0))).isSeparator())
            {
                bIsSeparator = true;
            }
            propertiesButton.setEnabled(!bIsSeparator);
        }
        else
            propertiesButton.setEnabled(false);
        boolean bCanMoveUp, bCanMoveDown;
        AbstractFSObject fsObj;
        bCanMoveUp = bCanMoveDown = true;
        Iterator<AbstractFSObject> iter = currentFSObjs.iterator();
        while (iter.hasNext())
        {
            fsObj = (iter.next());
            if (!currentDir.canMoveUp(fsObj))
                bCanMoveUp = false;
            if (!currentDir.canMoveDown(fsObj))
                bCanMoveDown = false;
        }
        moveUpButton.setEnabled(bCanMoveUp);
        moveDownButton.setEnabled(bCanMoveDown);
        // Fire the selection changed event.
        this.fireSelectionChanged(currentFSObjs);
    } // valueChanged

    /**
     * Gets the list of all currently selected file objects.
     * @return Selected file object list
     */
    public List<AbstractFSObject> getSelectedFiles ()
    {
        if (this.getCurrentFSObjects() == null)
            return new ArrayList<AbstractFSObject>();
        List<AbstractFSObject> fileObjs = new ArrayList<AbstractFSObject>(
        	this.getCurrentFSObjects().size());
        fileObjs.addAll(this.getCurrentFSObjects());
        return fileObjs;
    } // getSelectedFiles

    /**
     * Gets the file object that is immediately before the first selected
     * object.
     * @return The file object or <tt>null</tt>
     */
    public AbstractFSObject getPreviousFile ()
    {
        int index = listTable.getSelectedRow();
        if (index >= 0)
        {
            AbstractFSObject fsObj = m_fileTableModel.getFSObjectAt(index);
            if (fsObj == null && index >= 1)
                fsObj = m_fileTableModel.getFSObjectAt(index - 1);
            if (fsObj != null)
                return fsObj;
        }
        return null;
    } // getPreviousFile

    /**
     * Removes one object from the table.
     * @param fsObj - The file to remove
     */
    public void removeFSObject (AbstractFSObject fsObj)
    {
        m_fileTableModel.removeFSObject(fsObj);
    } // removeFSObject

    /**
     * This method is invoked by the file list table mouse listener to show the
     * file context menu.
     * @param x - Mouse x coordinate
     * @param y - Mouse y coordinate
     */
    public void showFileContextMenu (int x, int y)
    {
        DirectoryObject        currentDir;
        List<AbstractFSObject> currentFSObjs;
        AbstractFSObject       fsObj;
        FileObject             fileObj;
        Font                   fn;
        boolean                bIsExported, bHaveReference, bHaveSeparator;
        boolean                bCanMoveUp, bCanMoveDown;

        currentDir = this.getCurrentDirectory();
        currentFSObjs = this.getCurrentFSObjects();
        // Set the list selection if there is one selected object.
        if (currentFSObjs == null || currentFSObjs.size() <= 1)
        {
            int rowIndex = listTable.rowAtPoint(new Point(x, y));
            if (rowIndex >= 0)
            {
                listTable.setRowSelectionInterval(rowIndex, rowIndex);
                currentFSObjs = this.getCurrentFSObjects();
            }
        }
        if (currentDir == null || currentFSObjs == null ||
            currentFSObjs.size() == 0)
        {
            return;
        }
        if (m_filePopupMenu == null)
        {
            // Create the "File" context menu.
            m_filePopupMenu = new JPopupMenu("File Menu");
            m_fpProperties = new JMenuItem("Properties");
            m_fpProperties.addActionListener(this);
            m_fpOpen = new JMenuItem("Open");
            m_fpOpen.addActionListener(this);
            m_fpShowExported = new JMenuItem("Show Exported");
            m_fpShowExported.addActionListener(this);
            m_fpAssociate = new JMenuItem("Set File ...");
            m_fpAssociate.addActionListener(this);
            m_fpCut = new JMenuItem("Cut");
            m_fpCut.addActionListener(this);
            m_fpDelete = new JMenuItem("Delete");
            m_fpDelete.addActionListener(this);
            m_fpMoveUp = new JMenuItem("Move Up");
            m_fpMoveUp.addActionListener(this);
            m_fpMoveDown = new JMenuItem("Move Down");
            m_fpMoveDown.addActionListener(this);
            // Add the menu items.
            m_filePopupMenu.add(m_fpProperties);
            m_filePopupMenu.add(m_fpOpen);
            m_filePopupMenu.add(m_fpShowExported);
            JSeparator sep = new JSeparator();
            sep.setBackground(new java.awt.Color(255, 255, 255));
            sep.setForeground(new java.awt.Color(153, 153, 153));
            m_filePopupMenu.add(sep);
            m_filePopupMenu.add(m_fpAssociate);
            sep = new JSeparator();
            sep.setBackground(new java.awt.Color(255, 255, 255));
            sep.setForeground(new java.awt.Color(153, 153, 153));
            m_filePopupMenu.add(sep);
            m_filePopupMenu.add(m_fpCut);
            m_filePopupMenu.add(m_fpDelete);
            sep = new JSeparator();
            sep.setBackground(new java.awt.Color(255, 255, 255));
            sep.setForeground(new java.awt.Color(153, 153, 153));
            m_filePopupMenu.add(sep);
            m_filePopupMenu.add(m_fpMoveUp);
            m_filePopupMenu.add(m_fpMoveDown);
        }
        // Set bold font for the default action.
        fn = m_fpOpen.getFont();
        if (currentFSObjs.size() == 1)
        {
            if ((currentFSObjs.get(0) instanceof DirectoryObject) &&
                m_bDblClickOnDirOpens ||
                (currentFSObjs.get(0) instanceof FileObject) &&
                m_bDblClickOnFileOpens)
            {
                // Open is the default option.
                m_fpProperties.setFont(fn.deriveFont(Font.PLAIN));
                m_fpOpen.setFont(fn.deriveFont(Font.BOLD));
            }
            else
            {
                // Properties is the default option.
                m_fpProperties.setFont(fn.deriveFont(Font.BOLD));
                m_fpOpen.setFont(fn.deriveFont(Font.PLAIN));
            }
        }
        else
        {
            // Set both in plain font.
            m_fpProperties.setFont(fn.deriveFont(Font.PLAIN));
            m_fpOpen.setFont(fn.deriveFont(Font.PLAIN));
        }
        // Check the "m_currentFileObjs" contents.
        bIsExported = bHaveReference = bHaveSeparator = false;
        bCanMoveUp = bCanMoveDown = true;
        Iterator<AbstractFSObject> iter = currentFSObjs.iterator();
        while (iter.hasNext())
        {
            fsObj = (iter.next());
            if (fsObj instanceof FileObject)
            {
                fileObj = (FileObject) fsObj;
                if (fileObj.isSeparator())
                    bHaveSeparator = true;
                if (fileObj.getFileType() == FileObject.TYPE_IMAGE_PREVIEW &&
                    fileObj.isToExport())
                {
                    bIsExported = true;
                }
            }
            if (fsObj.isReference())
                bHaveReference = true;
            if (!currentDir.canMoveUp(fsObj))
                bCanMoveUp = false;
            if (!currentDir.canMoveDown(fsObj))
                bCanMoveDown = false;
        }
        // Enable the menu items based on the "m_currentFileObjs" contents.
        if (currentFSObjs.size() == 1)
        {
            m_fpOpen.setEnabled(!bHaveReference);
            m_fpShowExported.setEnabled(bIsExported);
            m_fpProperties.setEnabled(!bHaveSeparator);
            m_fpAssociate.setEnabled(bHaveReference && !bHaveSeparator);
        }
        else
        {
            m_fpOpen.setEnabled(false);
            m_fpShowExported.setEnabled(false);
            m_fpProperties.setEnabled(false);
            m_fpAssociate.setEnabled(false);
        }
        m_fpCut.setEnabled(!bHaveSeparator);
        m_fpMoveUp.setEnabled(bCanMoveUp);
        m_fpMoveDown.setEnabled(bCanMoveDown);
        // Show the context menu.
        m_filePopupMenu.show(listTable, x, y);
    } // showFileContextMenu

    /**
     * This method is invoked for the <b>File</b> context menu actions.
     * @param e - The event specifying which context menu item has been selected
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    public void actionPerformed (ActionEvent e)
    {
        Object                 evtSource         = e.getSource();
        List<AbstractFSObject> m_currentFileObjs = this.getCurrentFSObjects();

        if (evtSource == m_fpProperties)
            this.propertiesButtonActionPerformed(e);
        else if (evtSource == m_fpDelete)
            this.delButtonActionPerformed(e);
        else if (evtSource == m_fpMoveUp)
            this.moveUpButtonActionPerformed(e);
        else if (evtSource == m_fpMoveDown)
            this.moveDownButtonActionPerformed(e);
        else if (evtSource == m_fpOpen)
        {
            if (m_currentFileObjs.size() == 1)
            {
                if (m_currentFileObjs.get(0) instanceof DirectoryObject)
                {
                    // Open the directory.
                    DirectoryObject dirObj = (DirectoryObject)
                        m_currentFileObjs.get(0);
                    this.setCurrentDirectory(dirObj);
                    this.fireCurrentDirectoryChanged(dirObj);
                }
                else
                {
                    // Open the file.
                    ShellExec exec = new ShellExec(this.getFrame());
                    exec.openDocument((FileObject) m_currentFileObjs.get(0));
                }
            }
        }
        else if (evtSource == m_fpShowExported)
        {
            if (m_currentFileObjs.size() == 1 &&
                m_currentFileObjs.get(0) instanceof FileObject)
            {
                ExportedImageViewDialog dlg = new ExportedImageViewDialog(
                    (FileObject) (m_currentFileObjs.get(0)),
                    this.getFrame().getSettings(),
                    this.getFrame().getSeriesContainer(),
                    this.getFrame(), true);
                dlg.setVisible(true);
            }
        }
        else if (evtSource == m_fpCut)
        {
            // Cut the current file objects.
            this.getFrame().cut(m_currentFileObjs);
            pasteButton.setEnabled(this.getFrame().canPaste());
        }
        else if (evtSource == m_fpAssociate)
        {
            // Associate the current reference with a file.
            if (m_currentFileObjs.size() != 1 ||
                !(m_currentFileObjs.get(0) instanceof FileObject))
            {
                return;
            }
            AssociateDialog dia = new AssociateDialog
                (this.getFrame(), this.getFrame().getImageHolder(),
                (FileObject) (m_currentFileObjs.get(0)), true);
            if (!dia.showDialog())
                return;
            this.selectCurrentFiles(true, null);
        }
    } // actionPerformed

    /**
     * Renames the exported files for a file system object.
     * @param oldFsObj - Old file system object (before move or renaming)
     * @param newParent - New parent directory
     * @param newName - New file name
     */
    public void renameExportedFiles
        ( AbstractFSObject oldFsObj
        , DirectoryObject  newParent
        , String           newName
        )
    {
        this.getFrame().renameExportedFiles(oldFsObj, newParent, newName);
    } // renameExportedFiles

    /**
     * This method is invoked by the file list table mouse listener to show the
     * file properties dialog.
     */
    public void onDoubleClickListTable ()
    {
        List<AbstractFSObject> m_currentFileObjs = this.getCurrentFSObjects();
        if (m_currentFileObjs.size() != 1)
            return;
        AbstractFSObject currentFSObj = m_currentFileObjs.get(0);
        if (currentFSObj instanceof FileObject &&
            ((FileObject) currentFSObj).isSeparator())
        {
            return;
        }
        // If this is a directory, it should be opened.
        if (m_bDblClickOnDirOpens && (currentFSObj instanceof DirectoryObject))
        {
            DirectoryObject dirObj = (DirectoryObject) currentFSObj;
            this.setCurrentDirectory(dirObj);
            this.fireCurrentDirectoryChanged(dirObj);
            return;
        }
        if (m_bDblClickOnFileOpens && (currentFSObj instanceof FileObject))
        {
            ShellExec exec = new ShellExec(this.getFrame());
            exec.openDocument((FileObject) currentFSObj);
            return;
        }
        this.propertiesButtonActionPerformed(null);
    } // onDoubleClickListTable

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#loadSettings(com.dgrossmann.photo.settings.Settings)
     */
    public void loadSettings (Settings settings)
    {
        TableColumnModel cModel;
        TableColumn      col;
        String           valStr;
        StringTokenizer  tok;
        int              i;

        // Loads the column widths.
        cModel = listTable.getColumnModel();
        for (i = 1;; i++)
        {
            valStr = settings.get(CONT_COLUMN_POS + i);
            if (valStr == null || valStr.length() == 0)
                break;
            if (valStr.startsWith("-"))
                continue;
            col = cModel.getColumn(i-1);
            if (!col.getResizable())
                continue;
            tok = new StringTokenizer(valStr, ",");
            if (tok.countTokens() < 3)
                continue;
            try
            {
                col.setMinWidth(Integer.parseInt(tok.nextToken()));
                col.setMaxWidth(Integer.parseInt(tok.nextToken()));
                col.setPreferredWidth(Integer.parseInt(tok.nextToken()));
            }
            catch (Exception ignored)
            {
            }
        }
        // Set the double click options.
        m_bDblClickOnDirOpens = settings.getBoolean(DBL_CLICK_DIR_OPENS, false);
        m_bDblClickOnFileOpens = settings.getBoolean(DBL_CLICK_FILE_OPENS,
            false);
        // Load the preview settings.
        m_previewManager.loadSettings(settings);
        // Set the preview.
        m_bIsPreviewShowing = settings.getBoolean(CONT_SHOW_PREVIEW, false);
        this.getTitleLabel().setPanelComponents(aspectsPanel, contentsPanel,
            previewPanel, m_bIsPreviewShowing);
        this.getTitleLabel().addChangeListener(this);
        m_previewManager.setActive(m_bIsPreviewShowing);
    } // loadSettings

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#saveSettings(com.dgrossmann.photo.settings.Settings)
     */
    public void saveSettings (Settings settings)
    {
        // Saves the column widths.
        TableColumnModel cModel = listTable.getColumnModel();
        TableColumn col;
        for (int i = 1; i <= cModel.getColumnCount(); i++)
        {
            col = cModel.getColumn(i-1);
            if (!col.getResizable())
                settings.set(CONT_COLUMN_POS + i, "-");
            else
            {
                settings.set(CONT_COLUMN_POS + i,
                    Integer.toString(col.getMinWidth()) + "," +
                    col.getMaxWidth() + "," + col.getPreferredWidth());
            }
        }
        // Save the preview state.
        settings.setBoolean(CONT_SHOW_PREVIEW, m_bIsPreviewShowing);
        // Save the preview settings.
        m_previewManager.saveSettings(settings);
    } // saveSettings

    /**
     * @see java.awt.event.FocusListener#focusGained(java.awt.event.FocusEvent)
     */
    public void focusGained (FocusEvent e)
    {
        List<AbstractFSObject> ls = this.getCurrentFSObjects();
        if (ls != null && !ls.isEmpty())
            this.fireSelectionChanged(ls);
    } // focusGained

    /**
     * @see java.awt.event.FocusListener#focusLost(java.awt.event.FocusEvent)
     */
    public void focusLost (FocusEvent e)
    {
    } // focusLost

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents()
    {
        java.awt.GridBagConstraints gridBagConstraints;

        aspectsPanel = new javax.swing.JPanel();
        contentsPanel = new javax.swing.JPanel();
        buttonPanel = new javax.swing.JPanel();
        newObjPanel = new javax.swing.JPanel();
        newObjSelComboBox = new javax.swing.JComboBox();
        newObjButton = new javax.swing.JButton();
        separatorLabel = new javax.swing.JLabel();
        propertiesButton = new javax.swing.JButton();
        separatorLabel1 = new javax.swing.JLabel();
        pasteButton = new javax.swing.JButton();
        delButton = new javax.swing.JButton();
        separatorLabel2 = new javax.swing.JLabel();
        orderByFileNameButton = new javax.swing.JButton();
        separatorLabel3 = new javax.swing.JLabel();
        moveUpButton = new javax.swing.JButton();
        moveDownButton = new javax.swing.JButton();
        listScrollPane = new javax.swing.JScrollPane();
        listTable = new javax.swing.JTable();
        previewPanel = new javax.swing.JPanel();
        previewCenterPanel = new javax.swing.JPanel();
        jScrollPane1 = new javax.swing.JScrollPane();
        htmlPane = new javax.swing.JEditorPane();

        this.setLayout(new java.awt.BorderLayout());

        aspectsPanel.setLayout(new java.awt.CardLayout());

        contentsPanel.setLayout(new java.awt.GridBagLayout());

        buttonPanel.setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.LEFT, 5, 2));

        newObjPanel.setLayout(new java.awt.GridBagLayout());

        newObjSelComboBox.setToolTipText("Choose the kind of object to be created and press the \"New\" button");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        newObjPanel.add(newObjSelComboBox, gridBagConstraints);

        newObjButton.setText("New");
        newObjButton.setToolTipText("Creates an object of the kind selected in the combo box");
        newObjButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                ContentsPanel.this.newObjButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(0, 2, 0, 0);
        newObjPanel.add(newObjButton, gridBagConstraints);

        buttonPanel.add(newObjPanel);

        separatorLabel.setText(" ");
        separatorLabel.setMaximumSize(new java.awt.Dimension(5, 14));
        separatorLabel.setMinimumSize(new java.awt.Dimension(5, 14));
        separatorLabel.setPreferredSize(new java.awt.Dimension(5, 14));
        buttonPanel.add(separatorLabel);

        propertiesButton.setText("Properties");
        propertiesButton.setToolTipText("Displays the properties of the selected file or subgroup");
        propertiesButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                ContentsPanel.this.propertiesButtonActionPerformed(evt);
            }
        });
        buttonPanel.add(propertiesButton);

        separatorLabel1.setText(" ");
        separatorLabel1.setMaximumSize(new java.awt.Dimension(5, 14));
        separatorLabel1.setMinimumSize(new java.awt.Dimension(5, 14));
        separatorLabel1.setPreferredSize(new java.awt.Dimension(5, 14));
        buttonPanel.add(separatorLabel1);

        pasteButton.setText("Paste");
        pasteButton.setToolTipText("Pasts the cut file or directory into this directory");
        pasteButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                ContentsPanel.this.pasteButtonActionPerformed(evt);
            }
        });
        buttonPanel.add(pasteButton);

        delButton.setText("Delete");
        delButton.setToolTipText("Deletes the selected file or subgroup");
        delButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                ContentsPanel.this.delButtonActionPerformed(evt);
            }
        });
        buttonPanel.add(delButton);

        separatorLabel2.setText(" ");
        separatorLabel2.setMaximumSize(new java.awt.Dimension(5, 14));
        separatorLabel2.setMinimumSize(new java.awt.Dimension(5, 14));
        separatorLabel2.setPreferredSize(new java.awt.Dimension(5, 14));
        buttonPanel.add(separatorLabel2);

        orderByFileNameButton.setText("Order by File Name");
        orderByFileNameButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                ContentsPanel.this.orderByFileNameButtonActionPerformed(evt);
            }
        });
        buttonPanel.add(orderByFileNameButton);

        separatorLabel3.setText(" ");
        separatorLabel3.setMaximumSize(new java.awt.Dimension(5, 14));
        separatorLabel3.setMinimumSize(new java.awt.Dimension(5, 14));
        separatorLabel3.setPreferredSize(new java.awt.Dimension(5, 14));
        buttonPanel.add(separatorLabel3);

        moveUpButton.setText("Move Up");
        moveUpButton.setToolTipText("Moves up the selected file or subgroup");
        moveUpButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                ContentsPanel.this.moveUpButtonActionPerformed(evt);
            }
        });
        buttonPanel.add(moveUpButton);

        moveDownButton.setText("Move Down");
        moveDownButton.setToolTipText("Moves down the selected file or subgroup");
        moveDownButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                ContentsPanel.this.moveDownButtonActionPerformed(evt);
            }
        });
        buttonPanel.add(moveDownButton);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 0, 0, 0);
        contentsPanel.add(buttonPanel, gridBagConstraints);

        listTable.setModel(m_fileTableModel);
        listTable.setShowHorizontalLines(false);
        listTable.setShowVerticalLines(false);
        listScrollPane.setViewportView(listTable);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.weighty = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        contentsPanel.add(listScrollPane, gridBagConstraints);

        aspectsPanel.add(contentsPanel, "normal");

        previewPanel.setLayout(new java.awt.GridBagLayout());

        previewCenterPanel.setLayout(new java.awt.BorderLayout());

        jScrollPane1.setViewportView(htmlPane);

        previewCenterPanel.add(jScrollPane1, java.awt.BorderLayout.CENTER);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.weighty = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 6);
        previewPanel.add(previewCenterPanel, gridBagConstraints);

        aspectsPanel.add(previewPanel, "preview");

        this.add(aspectsPanel, java.awt.BorderLayout.CENTER);
    }// </editor-fold>//GEN-END:initComponents

    private void newObjButtonActionPerformed (java.awt.event.ActionEvent evt)//GEN-FIRST:event_newObjButtonActionPerformed
    {//GEN-HEADEREND:event_newObjButtonActionPerformed
        DirectoryObject currentDir = this.getCurrentDirectory();
        if (currentDir == null)
            return;
        String selObjType = (String) newObjSelComboBox.getSelectedItem();
        if (selObjType.equals(NEWITEM_FILE))
        {
            this.getFrame().newFile(currentDir);
        }
        else if (selObjType.equals(NEWITEM_SUBGROUP))
        {
            this.getFrame().newDirectory(currentDir);
        }
        else if (selObjType.equals(NEWITEM_REFERENCE))
        {
            // Create a new reference.
            FileObject fe = new FileObject("", currentDir);
            fe.setReference(true);
            this.getFrame().finishFileObj(fe);
        }
        else if (selObjType.equals(NEWITEM_SEPARATOR))
        {
            // Create a new separator.
            FileObject fe = new FileObject("", currentDir);
            fe.makeSeparator();
            this.getFrame().finishFileObj(fe);
        }
    }//GEN-LAST:event_newObjButtonActionPerformed

    private void propertiesButtonActionPerformed (java.awt.event.ActionEvent evt)//GEN-FIRST:event_propertiesButtonActionPerformed
    {//GEN-HEADEREND:event_propertiesButtonActionPerformed
        List<AbstractFSObject> m_currentFileObjs = this.getCurrentFSObjects();
        if (m_currentFileObjs == null || m_currentFileObjs.size() != 1)
            return;
        AbstractFSObject fsObj = (m_currentFileObjs.get(0));
        boolean oldToExport = fsObj.isToExport();
        int oldQuality = fsObj.getConversionQuality();
        File[] exFiles = ExportFactory.getExport(this.getFrame().getSettings(),
            this.getFrame().getSeriesContainer(), this).getExportedFiles(fsObj);
        PropertiesDialog dlg = new PropertiesDialog
            (this.getFrame().getImageHolder(), fsObj, exFiles, this.getFrame(),
            true);
        if (dlg.showDialog())
        {
            // Handle re-export of the selected file.
            if (fsObj instanceof FileObject &&
                !fsObj.isToExport())
            {
                try
                {
                    ExportFactory.getExport(this.getFrame().getSettings(),
                        this.getFrame().getSeriesContainer(), this).
                        deleteExportedFiles(fsObj, IWebExport.EXPORT_ALL);
                }
                catch (ExportException ignored)
                {
                }
            }
            else if (fsObj instanceof FileObject &&
                 ((!oldToExport && fsObj.isToExport()) ||
                  (oldQuality != fsObj.getConversionQuality())))
            {
                this.getFrame().startExport(fsObj, true, true, true);
            }
            // Refresh the table.
            this.selectCurrentFiles(true, null);
            this.firePropertiesChanged(fsObj);
        }
    }//GEN-LAST:event_propertiesButtonActionPerformed

    private void pasteButtonActionPerformed (java.awt.event.ActionEvent evt)//GEN-FIRST:event_pasteButtonActionPerformed
    {//GEN-HEADEREND:event_pasteButtonActionPerformed
        this.getFrame().paste();
        pasteButton.setEnabled(false);
    }//GEN-LAST:event_pasteButtonActionPerformed

    private void delButtonActionPerformed (java.awt.event.ActionEvent evt)//GEN-FIRST:event_delButtonActionPerformed
    {//GEN-HEADEREND:event_delButtonActionPerformed
        this.getFrame().delete(this.getCurrentFSObjects());
    }//GEN-LAST:event_delButtonActionPerformed

    private void moveUpButtonActionPerformed (java.awt.event.ActionEvent evt)//GEN-FIRST:event_moveUpButtonActionPerformed
    {//GEN-HEADEREND:event_moveUpButtonActionPerformed
        DirectoryObject            currentDir = this.getCurrentDirectory();
        List<AbstractFSObject>     currentFileObjs = this.getCurrentFSObjects();
        AbstractFSObject           fsObj;
        Iterator<AbstractFSObject> iter;
        boolean          bHaveDirs = false;

        if (currentDir == null || currentFileObjs.size() == 0)
            return;
        iter = currentFileObjs.iterator();
        while (iter.hasNext())
        {
            fsObj = (iter.next());
            // Move the object up.
            if (fsObj instanceof DirectoryObject)
            {
                bHaveDirs = true;
                currentDir.moveSubDirUp((DirectoryObject) fsObj);
            }
            else if (fsObj instanceof FileObject)
                currentDir.moveFileUp((FileObject) fsObj);
        }
        if (bHaveDirs)
            this.firePropertiesChanged(currentDir);
        this.selectCurrentFiles(true, null);
    }//GEN-LAST:event_moveUpButtonActionPerformed

    private void moveDownButtonActionPerformed (java.awt.event.ActionEvent evt)//GEN-FIRST:event_moveDownButtonActionPerformed
    {//GEN-HEADEREND:event_moveDownButtonActionPerformed
        DirectoryObject        currentDir = this.getCurrentDirectory();
        List<AbstractFSObject> currentFileObjs = this.getCurrentFSObjects();
        AbstractFSObject       fsObj;
        boolean                bHaveDirs = false;

        if (currentDir == null || currentFileObjs.size() == 0)
            return;
        for (int i = currentFileObjs.size()-1; i >= 0; i--)
        {
            fsObj = currentFileObjs.get(i);
            // Move the object down.
            if (fsObj instanceof DirectoryObject)
            {
                bHaveDirs = true;
                currentDir.moveSubDirDown((DirectoryObject) fsObj);
            }
            else if (fsObj instanceof FileObject)
                currentDir.moveFileDown((FileObject) fsObj);
        }
        if (bHaveDirs)
            this.firePropertiesChanged(currentDir);
        this.selectCurrentFiles(true, null);
    }//GEN-LAST:event_moveDownButtonActionPerformed

    private void orderByFileNameButtonActionPerformed (java.awt.event.ActionEvent evt)//GEN-FIRST:event_orderByFileNameButtonActionPerformed
    {//GEN-HEADEREND:event_orderByFileNameButtonActionPerformed
    	if (JOptionPane.showConfirmDialog(this,
    		"Permanently reorder the files by name?", "JPhoto-Explorer",
    		JOptionPane.YES_NO_OPTION) != JOptionPane.YES_OPTION)
    	{
    		return;
    	}
        DirectoryObject currentDir = this.getCurrentDirectory();
        if (currentDir == null)
            return;
        currentDir.orderFilesByFileName();
        this.firePropertiesChanged(currentDir);
        this.selectCurrentFiles(true, null);
    }//GEN-LAST:event_orderByFileNameButtonActionPerformed


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JPanel aspectsPanel;
    private javax.swing.JPanel buttonPanel;
    private javax.swing.JPanel contentsPanel;
    private javax.swing.JButton delButton;
    private javax.swing.JEditorPane htmlPane;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JScrollPane listScrollPane;
    private javax.swing.JTable listTable;
    private javax.swing.JButton moveDownButton;
    private javax.swing.JButton moveUpButton;
    private javax.swing.JButton newObjButton;
    private javax.swing.JPanel newObjPanel;
    private javax.swing.JComboBox newObjSelComboBox;
    private javax.swing.JButton orderByFileNameButton;
    private javax.swing.JButton pasteButton;
    private javax.swing.JPanel previewCenterPanel;
    private javax.swing.JPanel previewPanel;
    private javax.swing.JButton propertiesButton;
    private javax.swing.JLabel separatorLabel;
    private javax.swing.JLabel separatorLabel1;
    private javax.swing.JLabel separatorLabel2;
    private javax.swing.JLabel separatorLabel3;
    // End of variables declaration//GEN-END:variables
} // ContentsPanel
