/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
  Interface: IWebExport
    Created: 9 January, 2003
        $Id: IWebExport.java 76 2006-03-29 19:16:10Z Dirk $
  $Revision: 76 $
      $Date: 2006-03-29 21:16:10 +0200 (Mi, 29 Mrz 2006) $
    $Author: Dirk $
===============================================================================
*/

package com.dgrossmann.photo.webexport;

import java.awt.Component;
import java.io.File;

import com.dgrossmann.photo.dir.AbstractFSObject;
import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.SeriesContainer;
import com.dgrossmann.photo.settings.Settings;

/**
 * Interface to be implemented by Web exporters.
 */
public interface IWebExport
{
    public static final int EXPORT_NOTHING        = 0;
    public static final int EXPORT_ALL            = 1;
    public static final int EXPORT_MAIN_IMAGE     = 2;
    public static final int EXPORT_OVERVIEW_IMAGE = 3;

    /**
     * Gets a description of the method used by the exporter.
     * @return String description of this exporter
     */
    public String getDescription();

    /**
     * Sets the important objects.
     * @param settings - Settings object
     * @param sContainer - Series container object
     * @param parent - Parent component
     */
    public void setEnvironment
        ( Settings        settings
        , SeriesContainer sContainer
        , Component       parent
        );

    /**
     * Returns whether the exporter has all prerequisites installed (e.g., the
     * ImageMagick exporter requires ImageMagick and Java-Magick).
     * @return <tt>True</tt> if all prerequisites are available
     */
    public boolean isAvailable ();

    /**
     * Exports an image file to the Web export directory.
     * @param progress - Export progress holder object that holds the file
     * object to export and is used to display export status information
     * @param whatToExport - Can be one of <tt>EXPORT_ALL</tt>, export all image
     * forms; <tt>EXPORT_MAIN_IMAGE</tt>, export only the main image; or
     * <tt>EXPORT_OVERVIEW_IMAGE</tt>, export only the overview image.
     * @param bForce - <tt>True</tt> to force export even if the exported files
     * already exist and are younger than the originals
     * @throws ExportException on failure
     */
    public void export
        ( ExportProgress progress
        , int            whatToExport
        , boolean        bForce
        ) throws ExportException;

    /**
     * Copies a file to the Web export directory.
     * @param progress - Export progress holder object that holds the file
     * object to copy and is used to display export status information
     * @param bForce - <tt>True</tt> to force copy even if the exported file
     * already exists and is younger than the original
     * @throws ExportException on failure
     */
    public void copy
        ( ExportProgress progress
        , boolean        bForce
        ) throws ExportException;

    /**
     * Gets the exported files for a file if they exist (null otherwise).
     * @param fsObj - File system object whose exported files are needed
     * @return 2-element array containing the normal and small image version,
     * or <tt>null</tt> if there are no exported files
     */
    public File[] getExportedFiles (AbstractFSObject fsObj);

    /**
     * Renames the exported files when a file object is about to be renamed.
     * @param oldFsObj - Old file still having its old name
     * @param newParent - New parent directory or <tt>null</tt> if the parent
     * has not changed
     * @param newName - New name or <tt>null</tt> if the name has not changed
     * @throws ExportException
     */
    public void renameExportedFiles
        ( AbstractFSObject oldFsObj
        , DirectoryObject  newParent
        , String           newName
        ) throws ExportException;

    /**
     * Deletes the exported files for a file system object from the Web export
     * directory.
     * @param fsObj - File object whose exported files should be deleted
     * @param whatTodelete - Can be one of <tt>EXPORT_ALL</tt>, delete all
     * image forms; <tt>EXPORT_MAIN_IMAGE</tt>, delete only the main image; or
     * <tt>EXPORT_OVERVIEW_IMAGE</tt>, delete only the overview image.
     * @throws ExportException
     */
    public void deleteExportedFiles
        ( AbstractFSObject fsObj
        , int              whatTodelete
        ) throws ExportException;
} // IWebExport
