/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: JavaExporter
    Created: 9 January, 2003
        $Id: JavaExporter.java 158 2009-05-06 19:49:13Z dirk $
  $Revision: 158 $
      $Date: 2009-05-06 21:49:13 +0200 (Mi, 06 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.webexport;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.MediaTracker;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.io.File;

import javax.imageio.ImageIO;

import com.dgrossmann.photo.dir.FileObject;

/**
 * Exporter that only uses the Java image manipulation facilities to convert
 * the images.
 */
public class JavaExporter extends BaseExporter
{
    /** Constructor of a new <tt>JavaExporter</tt> instance. */
    public JavaExporter ()
    {
        super();
    } // JavaExporter

    /** Gets a description of the method used by the exporter. */
    public String getDescription ()
    {
        return "Internal conversion (for better quality, use ImageMagick)";
    } // getDescription

    /**
     * Returns whether the exporter has all prerequisites installed.
     * @return <tt>True</tt> because only Java is required
     */
    public boolean isAvailable ()
    {
        return true;
    } // isAvailable

    /* Private helper to scale an image and write it to the file. */
    private Image writeScaledImage
        ( Image     sourceImage
        , int       sourceImageType
        , Dimension size
        , int       quality
        , String    copyrightStr
        , File      destFile
        ) throws Exception
    {
        MediaTracker  track;
        Image         scaledImage;
        BufferedImage imageToWrite;

        scaledImage = sourceImage.getScaledInstance
            (size.width, size.height, Image.SCALE_SMOOTH);
        track = new MediaTracker(m_parent);
        track.addImage(scaledImage, 0);
        try
        {
            track.waitForAll();
        }
        catch (InterruptedException iexc)
        {
        }
        // Write the scaled image.
        imageToWrite = new BufferedImage
            (size.width, size.height, BufferedImage.TYPE_INT_RGB);
        Graphics2D gr = imageToWrite.createGraphics();
        boolean bReady = gr.drawImage(scaledImage, 0, 0, null);
        if (!bReady)
        {
            // Pfui Teufel (English: Hack).
            try
            {
                Thread.sleep(75);
            }
            catch (Exception sleepEx)
            {
            }
        }
        // Write the copyright string.
        if (copyrightStr != null)
        {
        	Font        fn;
        	FontMetrics fmetrics;
        	Rectangle2D box;

        	fn = gr.getFont();
        	fn = fn.deriveFont(Font.BOLD, 9.0F);
        	fmetrics = gr.getFontMetrics(fn);
        	box = fmetrics.getStringBounds(copyrightStr, gr);
        	// Black "shadow" text.
        	gr.setColor(Color.BLACK);
        	gr.drawString(copyrightStr,
        		(int) (size.width - box.getWidth() - 31),
        		(size.height - fmetrics.getDescent()));
        	// White "foreground" text.
        	gr.setColor(Color.WHITE);
        	gr.drawString(copyrightStr,
        		(int) (size.width - box.getWidth() - 32),
        		(size.height - fmetrics.getDescent() - 1));
        }
        // Write the image using the preferred method.
        try
        {
            ImageIO.write(imageToWrite, "jpeg", destFile);
            /* (This leads to large image sizes)
            Iterator writers = ImageIO.getImageWritersByFormatName("jpeg");
            ImageWriter writer = (ImageWriter) writers.next();
            ImageWriteParam param = writer.getDefaultWriteParam();
            float qual = (float) (quality / 100.0);
            param.setCompressionMode(ImageWriteParam.MODE_EXPLICIT);
            param.setCompressionQuality(qual);
            ImageOutputStream ios = ImageIO.createImageOutputStream(destFile);
            writer.setOutput(ios);
            writer.write(null, new IIOImage(imageToWrite, null, null), param);
            ios.close();
            */
        }
        catch (Exception e)
        {
            System.err.println("E: Exception during image write.\n   " + e);
            e.printStackTrace();
            // This does not honor the quality setting.
            ImageIO.write(imageToWrite, "jpeg", destFile);
        }
        // Free the resources and return.
        sourceImage.flush();
        imageToWrite.flush();
        return scaledImage;
    } // writeScaledImage

    /**
     * Method that really carries out the file export. The <tt>progress</tt>
     * object is filled with the output directory and the file names, and the
     * output directory exists.
     * @param fileObj - File object to export
     * @param progress - Filled progress object
     * @throws ExportException on failure
     */
    @Override
	protected void exportFile
        ( FileObject     fileObj
        , ExportProgress progress
        ) throws ExportException
    {
        BufferedImage origImage;
        Image         scaledImage;
        Dimension     size;
        File          f;
        int           imageType;

        // Read the original image. We let the caller cope with out-of-
        // memory errors.
        try
        {
            origImage = ImageIO.read(new File(fileObj.getFullPath()));
            // Get the image dimensions.
            imageType = origImage.getType();
            if (imageType <= 0)
                imageType = BufferedImage.TYPE_INT_ARGB;
            size = new Dimension();
            size.width = origImage.getWidth();
            size.height = origImage.getHeight();
            progress.setOriginalSize(size);
            this.fillImageSizes(progress);
            // Scale to the "main" image size.
            if (!progress.isMainUpToDate())
            {
                size = progress.getMainSize();
                f = new File(progress.getOutputDirectory() + File.separator
                    + progress.getMainFileName());
                scaledImage = this.writeScaledImage(origImage, imageType,
                    size, this.getConversionQuality(fileObj, true),
                    this.getCopyrightString(fileObj), f);
                origImage = null;
            }
            else
                scaledImage = origImage;
            progress.setMainReady(true);
            // Scale to the "overview" image size.
            if (!progress.isIndexUpToDate())
            {
                size = progress.getIndexSize();
                f = new File(progress.getOutputDirectory() + File.separator
                    + progress.getIndexFileName());
                this.writeScaledImage(scaledImage, imageType,
                    size, this.getConversionQuality(fileObj, false), null, f);
            }
            scaledImage = null;
            progress.setIndexReady(true);
        }
        catch (Exception exc)
        {
            throw new ExportException(fileObj, exc);
        }
        Runtime.getRuntime().gc();
    } // exportFile
} // JavaExporter
